<?php

namespace SignocoreToolkit\UI;

use SignocoreToolkit\Infrastructure\Traits\Singleton;
use SignocoreToolkit\Application\Constants;
use Imagick;
use Exception;

final class Assets
{
    use Singleton;

    protected function init(): void
    {
        add_action('wp_enqueue_scripts', [$this, 'enqueueFrontendScripts'], 99);
        add_action('parse_request', [$this, 'ensureFaviconExists'], 0);
        add_action('wp_footer', [$this, 'addLoader']);
    }

    public function enqueueFrontendScripts(): void
    {
        wp_deregister_script('jquery-migrate');
        wp_deregister_script('jquery');
        wp_register_script('jquery', includes_url('/js/jquery/jquery.min.js'), [], '3.7.1', true);
        wp_enqueue_script('jquery');

        wp_enqueue_style('signocore-toolkit-style', Constants::$assetsUri . '/css/signocore.min.css', [], Constants::$pluginVersion);
        wp_enqueue_script('signocore-toolkit-script', Constants::$assetsUri . '/js/signocore.min.js', [], Constants::$pluginVersion, true);
    }

    public function ensureFaviconExists(): void
    {
        if (wp_get_environment_type() !== 'production' || Constants::$isAdmin || defined('DOING_AJAX')) {
            return;
        }

        global $wp;

        $request = empty($wp->request) && !empty($wp->query_string) ? '?' . $wp->query_string : $wp->request;
        if ($request !== 'favicon.ico') {
            return;
        }

        $faviconPath = ABSPATH . 'favicon.ico';
        if (file_exists($faviconPath)) {
            return;
        }

        $iconUrl = get_site_icon_url(512);
        if (empty($iconUrl)) {
            $this->copyFallbackFavicon();
            return;
        }

        if (class_exists('Imagick')) {
            if (!function_exists('download_url')) {
                require_once ABSPATH . 'wp-admin/includes/file.php';
            }

            $tmpFile = download_url($iconUrl);
            if (is_wp_error($tmpFile)) {
                $this->copyFallbackFavicon();
                return;
            }

            try {
                $image = new Imagick($tmpFile);
                $image->setImageFormat('ico');
                $image->writeImage($faviconPath);
            } catch (Exception $e) {
                error_log('Failed to convert favicon: ' . $e->getMessage());
            }

            @unlink($tmpFile);
        }

        $this->copyFallbackFavicon();
    }

    private function copyFallbackFavicon(): void
    {
        $faviconPath = ABSPATH . 'favicon.ico';
        if (file_exists($faviconPath)) {
            return;
        }

        $defaultFavicon = SIGNOCORE_TOOLKIT_DIR . 'assets/img/favicon.ico';
        if (file_exists($defaultFavicon)) {
            copy($defaultFavicon, $faviconPath);
        }
    }

    public function addLoader(): void
    {
        if (Constants::$isAdmin) {
            return;
        }

        echo '<div class="signocore-loader hidden"><div class="signocore-spinner"></div><span class="signocore-loader-content">' . __("We're working - Please wait...", Constants::TEXT_DOMAIN) . '</span></div>';
    }
}
