<?php

declare(strict_types=1);

namespace SignocoreToolkit\Support;

use SignocoreToolkit\Application\Constants;

class UrlHelper
{
	public static function currentUrl(bool $keepQueryString = true): string
	{
		$uri = sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'] ?? '/'));
		$queryString = sanitize_text_field(wp_unslash($_SERVER['QUERY_STRING'] ?? ''));
		$permalinkStructure = get_option('permalink_structure', '/');

		if ($queryString) {
			$uri = explode('?', $uri)[0];
		}

		if (str_ends_with($permalinkStructure, '/')) {
			$uri = trailingslashit($uri);
		}

		$uri = htmlspecialchars(($uri ?: '/'), ENT_QUOTES);

		if ($keepQueryString && $queryString) {
			$uri .= '?' . $queryString;
		}

		return home_url($uri);
	}

	public static function isPaginated(): bool
	{
		global $wp_query;

		return $wp_query->max_num_pages > 1;
	}

	public static function isFirstPage(): bool
	{
		return !is_paged();
	}

	public static function isLastPage(): bool
	{
		if (is_paged()) {
			global $wp_query;

			return get_query_var('paged') === $wp_query->max_num_pages;
		}

		return false;
	}

	public static function maxPages(): int
	{
		global $wp_query;

		return intval($wp_query->max_num_pages);
	}

	/**
	 * Get URL content via file_get_contents.
	 *
	 * @deprecated 2.0.0 Use getUrlContentSecure() instead. This method disables SSL verification.
	 *
	 * @param string $url The URL to fetch content from
	 * @return string|false The URL content or false on failure
	 */
	public static function getUrlContent(string $url): string|false
	{
		if (defined('WP_DEBUG') && WP_DEBUG) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_trigger_error
			trigger_error(
				'UrlHelper::getUrlContent() is deprecated since version 2.0.0. Use UrlHelper::getUrlContentSecure() instead.',
				E_USER_DEPRECATED
			);
		}

		$verifySSL = apply_filters('signocore_toolkit_url_helper_verify_ssl', false);

		if ($verifySSL) {
			return self::getUrlContentSecure($url);
		}

		$context = stream_context_create([
			"ssl" => [
				"verify_peer" => false,       // SECURITY RISK: SSL verification disabled
				"verify_peer_name" => false,  // SECURITY RISK: Hostname verification disabled
			],
		]);

		return file_get_contents($url, false, $context);
	}

	/**
	 * Get URL content securely with SSL verification enabled.
	 *
	 * @param string $url The URL to fetch content from
	 * @param array<string, mixed> $args Optional arguments for wp_remote_get
	 * @return string|false The URL content or false on failure
	 */
	public static function getUrlContentSecure(string $url, array $args = []): string|false
	{
		$defaults = [
			'timeout' => 10,
			'sslverify' => true,
			'user-agent' => 'Signocore Toolkit/' . (defined('SIGNOCORE_TOOLKIT_FILE') ? Constants::$pluginVersion : '1.0'),
		];

		$args = wp_parse_args($args, $defaults);
		$response = wp_remote_get($url, $args);

		if (is_wp_error($response)) {
			if (defined('WP_DEBUG') && WP_DEBUG) {
				error_log(sprintf(
					'Signocore Toolkit: Failed to fetch URL %s: %s',
					$url,
					$response->get_error_message()
				));
			}

			return false;
		}

		$responseCode = wp_remote_retrieve_response_code($response);
		if ($responseCode !== 200) {
			if (defined('WP_DEBUG') && WP_DEBUG) {
				error_log(sprintf(
					'Signocore Toolkit: URL %s returned HTTP %d',
					$url,
					$responseCode
				));
			}

			return false;
		}

		return wp_remote_retrieve_body($response);
	}

	public static function isFeed(): bool
	{
		$isFeed = function_exists('is_feed') ? is_feed() : false;

		if (isset($_GET['feed']) && $_GET['feed'] !== '0' && $_GET['feed'] !== 'false') {
			$isFeed = true;
		}

		if (!$isFeed) {
			$uriBase = basename($_SERVER['REQUEST_URI'] ?? '');

			if ($uriBase === 'feed') {
				$queriedObject = get_queried_object();

				if (!$queriedObject) {
					$isFeed = true;
				}
			}
		}

		return $isFeed;
	}

	public static function isAmpPage(): bool
	{
		global $wp;

		if (function_exists('kadence_is_amp') && kadence_is_amp()) {
			return true;
		}

		if (function_exists('is_amp_endpoint') && is_amp_endpoint()) {
			return true;
		}

		if (isset($_GET['amp']) && $_GET['amp'] !== '0' && $_GET['amp'] !== 'false') {
			return true;
		}

		if (function_exists('amp_is_request') && amp_is_request()) {
			return true;
		}

		if (class_exists('AMP_Theme_Support') && method_exists('AMP_Theme_Support', 'is_amp_endpoint') && \AMP_Theme_Support::is_amp_endpoint()) { // @phpstan-ignore function.impossibleType
			return true;
		}

        return str_ends_with($wp->request, '/amp/');
	}

	/**
	 * @param array<string, string> $params
	 */
	public static function getRedirectsAdminUrl(string $type = '', array $params = []): string
	{
		$redirectsInSeparateMenu = (bool) get_option(Constants::OPTION_PREFIX . 'redirects_separate_menu', true);
		$slug = $type === '' || $type === '0' ? 'signocore-seo-redirects' : 'signocore-seo-redirects-' . $type;
		$url = $type === 'settings' ? admin_url('options-general.php?page=' . $slug) : admin_url('tools.php?page=' . $slug);

		if ($redirectsInSeparateMenu) {
			$url = admin_url('admin.php?page=' . $slug);
		}

		if ($params !== []) {
			$url .= '&' . http_build_query($params);
		}

		return esc_url_raw($url);
	}
}
