<?php

declare(strict_types=1);

namespace SignocoreToolkit\Support;

class TaxonomyHelper
{
    /** @var array<string, string>|null */
    private static ?array $taxonomiesCache = null;

    /**
     * @return array<string, string>
     */
    public static function getTaxonomies(): array
    {
        if (self::$taxonomiesCache !== null) {
            return self::$taxonomiesCache;
        }

        /** @var array<string, \WP_Taxonomy> $taxonomies */
        $taxonomies = get_taxonomies(['public' => true], 'objects');
        unset($taxonomies['post_format']);
        unset($taxonomies['fb_product_set']);
        unset($taxonomies['product_shipping_class']);

        $taxonomiesReturn = [];
        foreach ($taxonomies as $taxonomy) {
            $label = $taxonomy->label;

            $postType = $taxonomy->object_type[0] ?? null;
            if ($postType) {
                $postTypeObject = get_post_type_object($postType);
                if ($postTypeObject) {
                    $label .= sprintf(' (%s)', $postTypeObject->labels->name);
                }
            }

            $taxonomiesReturn[$taxonomy->name] = $label;
        }

        self::$taxonomiesCache = $taxonomiesReturn;

        return $taxonomiesReturn;
    }

    public static function termHasPosts(mixed $term, ?string $taxonomy = null): bool
    {
        global $wpdb;

        $termId = null;
        if ($term && is_numeric($term) && $taxonomy) {
            $termId = (int) $term;
        } elseif ($term && !is_wp_error($term) && isset($term->term_id)) {
            $termId = (int) $term->term_id;
            $taxonomy ??= $term->taxonomy ?? null;
        }

        if ($termId && $taxonomy) {
            $sql = $wpdb->prepare(
                "SELECT 1
				FROM {$wpdb->term_relationships} tr
				INNER JOIN {$wpdb->posts} p ON p.ID = tr.object_id
				INNER JOIN {$wpdb->term_taxonomy} tt ON tt.term_taxonomy_id = tr.term_taxonomy_id
				WHERE tt.term_id = %d
				AND tt.taxonomy = %s
				AND p.post_status = 'publish'
				LIMIT 1",
                $termId,
                $taxonomy
            );

            return (bool) $wpdb->get_var($sql);
        }

        return false;
    }

    public static function getPrimaryCategory(int $postId): mixed
    {
        $primaryCategory = null;
        $primaryCategoryId = get_post_meta($postId, '_primary_category', true);

        if (empty($primaryCategoryId)) {
            $categories = get_the_category($postId);

            if (!empty($categories)) {
                $primaryCategory = reset($categories);
            }
        } else {
            $primaryCategory = get_term($primaryCategoryId);
        }

        return $primaryCategory;
    }

    public static function isRealArchive(): bool
    {
        if (is_archive() && !is_search() && !is_date() && !is_post_type_archive()) {
            $object = get_queried_object();
            if ($object instanceof \WP_Term) {
                return true;
            }
        }

        return false;
    }

    public static function shouldReplaceContent(?string $template = null): bool
    {
        if ($template !== get_template_directory() . '/template-parts/content/error.php') {
            return false;
        }

        if (!self::isRealArchive()) {
            return false;
        }

        $object = get_queried_object();
        if ($object && isset($object->term_id, $object->taxonomy)) {
            return !self::termHasPosts($object->term_id, $object->taxonomy);
        }

        return false;
    }

    public static function hasContent(?object $object = null): bool
    {
        $object = $object ?: get_queried_object();

        if (!$object instanceof \WP_Term) {
            return false;
        }

        $archiveDescription = is_author()
            ? get_user_meta($object->term_id, 'intro_text', true)
            : get_term_meta($object->term_id, 'intro_text', true);

        return !empty($archiveDescription);
    }

    public static function hasRelatedContent(?object $object = null): bool
    {
        $object = $object ?: get_queried_object();

        if (!$object instanceof \WP_Term) {
            return false;
        }

        if (is_tag()) {
            if (defined('SIGNOCORE_GLOSSARY_VERSION')) {
                $glossaryTermsByTitle = GlossaryHelper::termsByTitle($object->name);

                if ($glossaryTermsByTitle !== []) {
                    return true;
                }

                $glossaryTermsByTag = get_posts([
                    'post_type' => 'wpsseo_term',
                    'post_status' => 'publish',
                    'numberposts' => 1,
                    'return' => 'ids',
                    'suppress_filters' => false,
                    'no_found_rows' => true,
                    'ignore_sticky_posts' => true,
                    'update_post_meta_cache' => false,
                    'update_post_term_cache' => false,
                    'tax_query' => [
                        [
                            'taxonomy' => 'post_tag',
                            'field' => 'slug',
                            'terms' => [$object->slug],
                        ]
                    ]
                ]);

                if (!empty($glossaryTermsByTag)) {
                    return true;
                }
            }

            if (defined('SIGNOCORE_FAQ_VERSION')) {
                $faqsByTag = get_posts([
                    'post_type' => 'wpsseo_faq',
                    'post_status' => 'publish',
                    'numberposts' => 1,
                    'return' => 'ids',
                    'suppress_filters' => false,
                    'no_found_rows' => true,
                    'ignore_sticky_posts' => true,
                    'update_post_meta_cache' => false,
                    'update_post_term_cache' => false,
                    'tax_query' => [
                        [
                            'taxonomy' => 'post_tag',
                            'field' => 'slug',
                            'terms' => [$object->slug],
                        ]
                    ]
                ]);

                if (!empty($faqsByTag)) {
                    return true;
                }
            }
        }

        return false;
    }
}
