<?php

declare(strict_types=1);

namespace SignocoreToolkit\Support;

use SignocoreToolkit\Application\Constants;
use WP_Post;

/**
 * Reusable social share functionality with caching support.
 */
class SocialShareHelper
{
	/**
	 * Cache duration for social share HTML (1 month).
	 */
	private const CACHE_DURATION = MONTH_IN_SECONDS;

	/**
	 * Generate social share buttons HTML for a post.
	 *
	 * @param WP_Post|int|null $post Post object or ID (null for current post)
	 * @param string $class Additional CSS class for the container
	 * @param string|null $title Custom title (null for auto-generated based on post type)
	 * @param bool $useCache Whether to use transient caching (default: true)
	 * @return string HTML markup for social share buttons
	 */
	public static function generateShareButtons(
		WP_Post|int|null $post = null,
		string $class = '',
		?string $title = null,
		bool $useCache = true
	): string {
		// Get post object
		if (is_numeric($post)) {
			$post = get_post($post);
		} elseif (!$post instanceof \WP_Post) {
			$post = get_post();
		}

		if (!$post instanceof WP_Post) {
			return '';
		}

		// Try to get from cache
		if ($useCache) {
			$cacheKey = self::getCacheKey($post->ID, $class);
			$cached = get_transient($cacheKey);

			if ($cached !== false) {
				return $cached;
			}
		}

		$shareUrls = self::generateShareUrls($post);

		if ($title === null) {
			$title = self::getDefaultTitle($post);
		}

		$html = self::buildShareHtml($shareUrls, $title, $class);
		$html = apply_filters('signocore_social_share', $html, $post);

		if ($useCache) {
			set_transient($cacheKey, $html, self::CACHE_DURATION);
		}

		return $html;
	}

	/**
     * Generate share URLs for various social platforms.
     *
     * @param WP_Post $post The post object
     * @return array<string, string|bool> Array of platform => URL mappings
     */
    private static function generateShareUrls(WP_Post $post): array
	{
		$permalink = Constants::$currentUrl;
		$encodedLink = urlencode($permalink);

		$postImage = wp_get_attachment_image_url((int) get_post_thumbnail_id($post->ID), 'large');
		$encodedImage = urlencode($postImage ?: '');

		return [
			'facebook' => 'https://www.facebook.com/sharer.php?u=' . $encodedLink,
			'twitter' => 'https://x.com/intent/tweet?url=' . $encodedLink,
			'linkedin' => 'https://www.linkedin.com/sharing/share-offsite/?url=' . $encodedLink,
			'pinterest' => 'https://pinterest.com/pin/create/button/?description=' . $encodedLink . '&media=' . $encodedImage,
			'email' => 'mailto:?subject=' . rawurlencode($post->post_title) . '&body=' . $encodedLink,
			'has_image' => !empty($postImage),
		];
	}

	/**
	 * Get default title based on post type.
	 *
	 * @param WP_Post $post The post object
	 * @return string Localized title text
	 */
	private static function getDefaultTitle(WP_Post $post): string
	{
		return match ($post->post_type) {
			'product' => __('Share this product', Constants::TEXT_DOMAIN),
			'post' => __('Share this post', Constants::TEXT_DOMAIN),
			default => __('Share this page', Constants::TEXT_DOMAIN),
		};
	}

	/**
     * Build social share HTML markup.
     *
     * @param array<string, mixed> $urls Share URLs array
     * @param string $title Title text
     * @param string $class Additional CSS class
     * @return string HTML markup
     */
    private static function buildShareHtml(array $urls, string $title, string $class): string
	{
		$containerClass = 'social-share' . ($class !== '' && $class !== '0' ? ' ' . esc_attr($class) : '');

		$html = '<div class="' . $containerClass . '">';
		$html .= '<strong>' . esc_html($title) . ':</strong>';

		$html .= self::buildShareLink(
			'facebook',
			$urls['facebook'],
			__('Share on Facebook', Constants::TEXT_DOMAIN)
		);

		$html .= self::buildShareLink(
			'twitter',
			$urls['twitter'],
			__('Share on X', Constants::TEXT_DOMAIN)
		);

		$html .= self::buildShareLink(
			'linkedin',
			$urls['linkedin'],
			__('Share on LinkedIn', Constants::TEXT_DOMAIN)
		);

		if ($urls['has_image']) {
			$html .= self::buildShareLink(
				'pinterest',
				$urls['pinterest'],
				__('Share on Pinterest', Constants::TEXT_DOMAIN)
			);
		}

		$html .= '<a class="mail" href="' . esc_url($urls['email']) . '" aria-label="' . esc_attr__('Share through Email', Constants::TEXT_DOMAIN) . '">';
		$html .= '<i class="si si-mail" aria-hidden="true"></i>';
		$html .= '</a>';

		return $html . '</div>';
	}

	/**
	 * Build individual share link with popup JavaScript.
	 *
	 * @param string $platform Platform name
	 * @param string $url Share URL
	 * @param string $ariaLabel Accessibility label
	 * @param string|null $popupDimensions Custom popup dimensions
	 * @return string HTML for share link
	 */
	private static function buildShareLink(
		string $platform,
		string $url,
		string $ariaLabel,
		?string $popupDimensions = null
	): string {
		$dimensions = $popupDimensions ?? match ($platform) {
			'facebook' => 'width=500, height=700',
			'twitter' => 'width=480, height=420',
			'linkedin' => 'width=480, height=400',
			'pinterest' => 'width=720, height=640',
			default => 'width=600, height=500',
		};

		$html = '<a class="' . esc_attr($platform) . '" ';
		$html .= 'onclick="window.open(\'' . esc_js($url) . "', 'newwindow', '" . esc_js($dimensions) . '\'); return false" ';
		$html .= 'href="' . esc_url($url) . '" ';
		$html .= 'target="_blank" ';
		$html .= 'aria-label="' . esc_attr($ariaLabel) . '">';
		$html .= '<i class="si si-' . esc_attr($platform) . '" aria-hidden="true"></i>';

		return $html . '</a>';
	}

	/**
	 * Generate cache key for social share HTML.
	 *
	 * @param int $postId Post ID
	 * @param string $class Additional class
	 * @return string Cache key
	 */
	private static function getCacheKey(int $postId, string $class = ''): string
	{
		$suffix = $class !== '' && $class !== '0' ? '_' . sanitize_key($class) : '';
		return 'social_share_' . $postId . $suffix;
	}

	/**
     * Clear social share cache for a specific post.
     *
     * @param int $postId Post ID
     */
    public static function clearCache(int $postId): void
	{
		delete_transient('social_share_' . $postId);
		delete_transient('social_share_' . $postId . '_product-social-share');
	}

	/**
     * Clear social share cache for multiple posts.
     *
     * @param array<int, int> $postIds Array of post IDs
     */
    public static function clearCacheBulk(array $postIds): void
	{
		foreach ($postIds as $postId) {
			self::clearCache((int) $postId);
		}
	}
}
