<?php

declare(strict_types=1);

namespace SignocoreToolkit\Support;

class PostTypeHelper
{
	/**
	 * Cache duration for post type queries (1 hour).
	 */
	private const CACHE_DURATION = HOUR_IN_SECONDS;

	/**
	 * Get WooCommerce page IDs with caching support.
	 *
	 * @return array<int, int> List of WooCommerce page IDs
	 */
	public static function woocommerceIds(): array
	{
		$currentLang = defined('ICL_LANGUAGE_CODE') ? ICL_LANGUAGE_CODE : null;
		$cacheKey = 'sctk_wc_page_ids_' . ($currentLang ?? 'default');

		// Try to get from cache first
		$cachedIds = get_transient($cacheKey);
		if ($cachedIds !== false) {
			return $cachedIds;
		}

		$woocommerceIds = [];
		$woocommercePages = [
			'woocommerce_cart_page_id',
			'woocommerce_checkout_page_id',
			'woocommerce_pay_page_id',
			'woocommerce_thanks_page_id',
			'woocommerce_myaccount_page_id',
			'woocommerce_edit_address_page_id',
			'woocommerce_view_order_page_id',
			'woocommerce_terms_page_id',
			'woocommerce_refund_returns_page_id',
		];

		foreach ($woocommercePages as $optionName) {
			$pageId = get_option($optionName, 0);

			if ($pageId && $currentLang) {
				$translatedId = apply_filters('wpml_object_id', $pageId, 'page', false, $currentLang);

				if ($translatedId) {
					$woocommerceIds[] = (int) $translatedId;
				}
			} elseif ($pageId) {
				$woocommerceIds[] = (int) $pageId;
			}
		}

		// Cache the result
		set_transient($cacheKey, $woocommerceIds, self::CACHE_DURATION);

		return $woocommerceIds;
	}

	/** @var array<string, string>|null */
	private static ?array $postTypesCache = null;

	/**
	 * @return array<string, string>
	 */
	public static function getPostTypes(): array
	{
		if (self::$postTypesCache !== null) {
			return self::$postTypesCache;
		}

		$postTypesInitial = get_post_types(['public' => true], 'objects');
		$postTypes = [];

		unset($postTypesInitial['e-floating-buttons']);
		unset($postTypesInitial['elementor_library']);

		foreach ($postTypesInitial as $postType) {
			$postTypes[$postType->name] = $postType->label;
		}

		self::$postTypesCache = $postTypes;

		return $postTypes;
	}

	/**
	 * Get post IDs by post type, excluding noindex posts and WooCommerce pages.
	 *
	 * @param string $postType The post type to query (default: 'post')
	 * @return array<int, string> List of post IDs
	 */
	public static function getPostTypeIds(string $postType = 'post'): array
	{
		$cacheKey = 'sctk_post_ids_' . $postType;

		// Try to get from cache first
		$cachedIds = get_transient($cacheKey);
		if ($cachedIds !== false) {
			return $cachedIds;
		}

		global $wpdb;

		$postTypeIds = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT p.`ID`
				FROM `{$wpdb->posts}` p
				LEFT JOIN `{$wpdb->postmeta}` pm
					ON p.`ID` = pm.`post_id`
					AND pm.`meta_key` = '_genesis_noindex'
					AND pm.`meta_value` = '1'
				WHERE p.`post_type` = %s
					AND p.`post_status` = 'publish'
					AND pm.`post_id` IS NULL",
				$postType
			)
		);

		// Exclude WooCommerce pages
		$woocommerceIds = self::woocommerceIds();
		if ($woocommerceIds !== []) {
			$postTypeIds = array_diff($postTypeIds, $woocommerceIds);
		}

		// Ensure sequential numeric keys after array_diff
		$postTypeIds = array_values($postTypeIds);

		// Cache the result
		set_transient($cacheKey, $postTypeIds, self::CACHE_DURATION);

		return $postTypeIds;
	}


	/**
     * @param array<int, array<int, \WP_Post>> $children
     * @param array<int, string> $return
     */
    private static function getPostsRecursive(int $parentId, array $children, array &$return, int $level = 0): void
	{
		$level++;
		$padding = str_repeat('&nbsp;', $level * 3);

		if (!empty($children[$parentId])) {
			foreach ($children[$parentId] as $child) {
				$return[$child->ID] = $padding . esc_html($child->post_title);
				self::getPostsRecursive($child->ID, $children, $return, $level);
			}
		}
	}

	/**
	 * @param array<string, mixed> $args
	 * @return array<int, string>
	 */
	public static function dropdownPages(array $args = []): array
	{
		$defaults = [
			'posts_per_page'          => -1,
			'offset'                  => 0,
			'orderby'                 => 'menu_order',
			'order'                   => 'ASC',
			'include'                 => '',
			'exclude'                 => '',
			'meta_key'                => '',
			'meta_value'              => '',
			'post_type'               => 'page',
			'post_status'             => 'publish',
			'suppress_filters'        => false,
			'hierarchical'            => true,
			'no_found_rows'           => true,
			'ignore_sticky_posts'     => true,
			'update_post_meta_cache'  => false,
			'update_post_term_cache'  => false,
		];

		/** @var \WP_Post[] $posts */
		$posts = get_posts(array_merge($defaults, $args));

		$children = [];
		$return = [];

		foreach ($posts as $post) {
			if ($post->post_parent == 0) {
				$return[$post->ID] = esc_html($post->post_title);
			} else {
				$children[$post->post_parent][] = $post;
			}
		}

		foreach (array_keys($return) as $parentId) {
			self::getPostsRecursive($parentId, $children, $return);
		}

		return $return;
	}
}
