<?php

declare(strict_types=1);

namespace SignocoreToolkit\Support;

/**
 * Centralized detection of installed and active WordPress plugins.
 *
 * @package SignocoreToolkit\Support
 * @since 1.0.0
 */
class PluginCompatibility
{
	private const CACHE_SEO = 'seo';

	private const CACHE_WOOCOMMERCE = 'woocommerce';

	private const CACHE_FAQ = 'faq';

	private const CACHE_GLOSSARY = 'glossary';

	private const CACHE_KADENCE = 'kadence';

	private const CACHE_YOAST = 'yoast';

	private const CACHE_ACF = 'acf';

	/** @var array<string, bool> */
	private static array $cache = [];

	/**
	 * @return bool True if Signocore SEO is installed and active
	 */
	public static function isSeoActive(): bool
	{
		return self::detect(self::CACHE_SEO, fn(): bool => defined('SIGNOCORE_SEO_FILE'));
	}

	/**
	 * @return bool True if Signocore SEO plugin files exist but the plugin is not active
	 */
	public static function isSeoInstalled(): bool
	{
		if (self::isSeoActive()) {
			return false;
		}

		return file_exists(WP_PLUGIN_DIR . '/signocore-seo/signocore-seo.php');
	}

	/**
	 * @return bool True if WooCommerce is installed and active
	 */
	public static function isWooCommerceActive(): bool
	{
		return self::detect(self::CACHE_WOOCOMMERCE, fn(): bool => class_exists('WooCommerce'));
	}

	/**
	 * @return bool True if Signocore FAQ is installed and active
	 */
	public static function isFaqActive(): bool
	{
		return self::detect(self::CACHE_FAQ, fn(): bool => defined('SIGNOCORE_FAQ_VERSION'));
	}

	/**
	 * @return bool True if Signocore Glossary is installed and active
	 */
	public static function isGlossaryActive(): bool
	{
		return self::detect(self::CACHE_GLOSSARY, fn(): bool => defined('SIGNOCORE_GLOSSARY_VERSION'));
	}

	/**
	 * @return bool True if Kadence theme is active
	 */
	public static function isKadenceThemeActive(): bool
	{
		return self::detect(self::CACHE_KADENCE, fn(): bool => class_exists('Kadence\Theme'));
	}

	/**
	 * @return bool True if Yoast SEO is installed and active
	 */
	public static function isYoastActive(): bool
	{
		return self::detect(self::CACHE_YOAST, fn(): bool => class_exists('WPSEO_Options'));
	}

	/**
	 * @return bool True if ACF is installed and active
	 */
	public static function isAcfActive(): bool
	{
		return self::detect(self::CACHE_ACF, fn(): bool => function_exists('acf'));
	}

	/**
	 * @return array<string, bool> Plugin statuses
	 */
	public static function getActiveSignocorePlugins(): array
	{
		return [
			'seo' => self::isSeoActive(),
			'faq' => self::isFaqActive(),
			'glossary' => self::isGlossaryActive(),
		];
	}

	/**
	 * @return array<string, bool> Plugin statuses
	 */
	public static function getActiveThirdPartyPlugins(): array
	{
		return [
			'woocommerce' => self::isWooCommerceActive(),
			'yoast' => self::isYoastActive(),
			'acf' => self::isAcfActive(),
		];
	}

	/**
	 * @return array<string, bool> Theme statuses
	 */
	public static function getActiveThemes(): array
	{
		return [
			'kadence' => self::isKadenceThemeActive(),
		];
	}

	/**
	 * Clear the detection cache.
	 */
	public static function clearCache(): void
	{
		self::$cache = [];
	}

	/**
	 * Generic detection helper with caching.
	 *
	 * @param string $cacheKey Unique identifier for this check
	 * @param callable $detector Function that returns true if plugin is active
	 * @return bool True if plugin is active
	 */
	private static function detect(string $cacheKey, callable $detector): bool
	{
		if (!isset(self::$cache[$cacheKey])) {
			self::$cache[$cacheKey] = $detector();
		}

		return self::$cache[$cacheKey];
	}

	/**
	 * Get plugin version if available.
	 *
	 * @param string $plugin Plugin identifier (faq, glossary, woocommerce)
	 * @return string|null Version number or null
	 */
	public static function getPluginVersion(string $plugin): ?string
	{
		return match ($plugin) {
			'seo' => defined('SIGNOCORE_SEO_FILE') ? SIGNOCORE_SEO_FILE : null,
			'faq' => defined('SIGNOCORE_FAQ_FILE') ? SIGNOCORE_FAQ_FILE : null,
			'glossary' => defined('SIGNOCORE_GLOSSARY_FILE') ? SIGNOCORE_GLOSSARY_FILE : null,
			'woocommerce' => defined('WC_VERSION') ? WC_VERSION : null,
			'yoast' => defined('WPSEO_VERSION') ? WPSEO_VERSION : null,
			default => null,
		};
	}
}
