<?php

declare(strict_types=1);

namespace SignocoreToolkit\Support;

/**
 * Centralized logging system with PSR-3 compatible interface.
 *
 * @package SignocoreToolkit\Support
 * @since 1.0.0
 */
class Logger
{
	/** @var string */
	public const EMERGENCY = 'EMERGENCY';

	/** @var string */
	public const ALERT = 'ALERT';

	/** @var string */
	public const CRITICAL = 'CRITICAL';

	/** @var string */
	public const ERROR = 'ERROR';

	/** @var string */
	public const WARNING = 'WARNING';

	/** @var string */
	public const NOTICE = 'NOTICE';

	/** @var string */
	public const INFO = 'INFO';

	/** @var string */
	public const DEBUG = 'DEBUG';

	/** @var string */
	private const LOG_PREFIX = 'Signocore Toolkit:';

	/**
	 * @param string $message Log message
	 * @param array<string, mixed> $context Additional context data
	 */
	public static function emergency(string $message, array $context = []): void
	{
		self::log(self::EMERGENCY, $message, $context);
	}

	/**
	 * @param string $message Log message
	 * @param array<string, mixed> $context Additional context data
	 */
	public static function alert(string $message, array $context = []): void
	{
		self::log(self::ALERT, $message, $context);
	}

	/**
	 * @param string $message Log message
	 * @param array<string, mixed> $context Additional context data
	 */
	public static function critical(string $message, array $context = []): void
	{
		self::log(self::CRITICAL, $message, $context);
	}

	/**
	 * @param string $message Log message
	 * @param array<string, mixed> $context Additional context data
	 */
	public static function error(string $message, array $context = []): void
	{
		self::log(self::ERROR, $message, $context);
	}

	/**
	 * @param string $message Log message
	 * @param array<string, mixed> $context Additional context data
	 */
	public static function warning(string $message, array $context = []): void
	{
		self::log(self::WARNING, $message, $context);
	}

	/**
	 * @param string $message Log message
	 * @param array<string, mixed> $context Additional context data
	 */
	public static function notice(string $message, array $context = []): void
	{
		self::log(self::NOTICE, $message, $context);
	}

	/**
	 * @param string $message Log message
	 * @param array<string, mixed> $context Additional context data
	 */
	public static function info(string $message, array $context = []): void
	{
		self::log(self::INFO, $message, $context);
	}

	/**
	 * @param string $message Log message
	 * @param array<string, mixed> $context Additional context data
	 */
	public static function debug(string $message, array $context = []): void
	{
		self::log(self::DEBUG, $message, $context);
	}

	/**
	 * Core logging method.
	 *
	 * @param string $level Log level
	 * @param string $message Log message
	 * @param array<string, mixed> $context Additional context data
	 */
	public static function log(string $level, string $message, array $context = []): void
	{
		if (!self::shouldLog($level)) {
			return;
		}

		$formattedMessage = self::formatMessage($level, $message, $context);
		error_log($formattedMessage);
	}

	/**
	 * Check if logging should occur for this level.
	 *
	 * @param string $level Log level
	 * @return bool True if should log
	 */
	private static function shouldLog(string $level): bool
	{
		$criticalLevels = [
			self::EMERGENCY,
			self::ALERT,
			self::CRITICAL,
			self::ERROR,
		];

		if (in_array($level, $criticalLevels, true)) {
			return true;
		}

		return defined('WP_DEBUG') && WP_DEBUG;
	}

	/**
	 * Format log message with timestamp, level, and context.
	 *
	 * @param string $level Log level
	 * @param string $message Log message
	 * @param array<string, mixed> $context Context data
	 * @return string Formatted log message
	 */
	private static function formatMessage(string $level, string $message, array $context): string
	{
		$timestamp = current_time('Y-m-d H:i:s');
		$formatted = sprintf('[%s] [%s] %s %s', $timestamp, $level, self::LOG_PREFIX, $message);

		if ($context !== []) {
			$formatted .= ' ' . wp_json_encode($context, JSON_UNESCAPED_SLASHES);
		}

		return $formatted;
	}

	/**
	 * Log an exception with full stack trace.
	 *
	 * @param \Throwable $exception Exception to log
	 * @param string $level Log level (default: ERROR)
	 * @param array<string, mixed> $context Additional context data
	 */
	public static function exception(\Throwable $exception, string $level = self::ERROR, array $context = []): void
	{
		$message = sprintf(
			'%s: %s in %s:%d',
			$exception::class,
			$exception->getMessage(),
			$exception->getFile(),
			$exception->getLine()
		);

		$context['exception'] = [
			'class' => $exception::class,
			'message' => $exception->getMessage(),
			'code' => $exception->getCode(),
			'file' => $exception->getFile(),
			'line' => $exception->getLine(),
			'trace' => $exception->getTraceAsString(),
		];

		self::log($level, $message, $context);
	}

	/**
	 * Log to a specific file in the uploads directory.
	 *
	 * @param string $filename Log filename (without .log extension)
	 * @param string $level Log level
	 * @param string $message Log message
	 * @param array<string, mixed> $context Context data
	 * @return bool True if logged successfully
	 */
	public static function logToFile(string $filename, string $level, string $message, array $context = []): bool
	{
		if (!defined('WP_DEBUG') || !WP_DEBUG) {
			return false;
		}

		$uploadDir = wp_upload_dir();
		$logDir = $uploadDir['basedir'] . '/signocore-toolkit-logs';

		if (!file_exists($logDir)) {
			wp_mkdir_p($logDir);
		}

		if (!is_writable($logDir)) {
			return false;
		}

		$formattedMessage = self::formatMessage($level, $message, $context);
		$logFile = $logDir . '/' . $filename . '.log';
		return (bool) file_put_contents($logFile, $formattedMessage . PHP_EOL, FILE_APPEND);
	}

	/**
	 * Get recent log entries from a log file.
	 *
	 * @param string $filename Log filename (without .log extension)
	 * @param int $lines Number of lines to retrieve
	 * @return array<string> Array of log lines
	 */
	public static function getRecentLogs(string $filename, int $lines = 50): array
	{
		$uploadDir = wp_upload_dir();
		$logFile = $uploadDir['basedir'] . '/signocore-toolkit-logs/' . $filename . '.log';

		if (!file_exists($logFile)) {
			return [];
		}

		$fileLines = file($logFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);

		if ($fileLines === false) {
			return [];
		}

		return array_slice($fileLines, -$lines);
	}

	/**
	 * Clear a specific log file.
	 *
	 * @param string $filename Log filename (without .log extension)
	 * @return bool True if deleted successfully
	 */
	public static function clearLog(string $filename): bool
	{
		$uploadDir = wp_upload_dir();
		$logFile = $uploadDir['basedir'] . '/signocore-toolkit-logs/' . $filename . '.log';

		if (!file_exists($logFile)) {
			return false;
		}

		return unlink($logFile);
	}
}
