<?php

declare(strict_types=1);

namespace SignocoreToolkit\Support;

use SignocoreToolkit\Application\Constants;

class LocaleHelper
{
	public static function getLocaleFromLanguageCode(string $code): string
	{
		$map = [
			'aa' => 'aa_DJ',
			'ab' => 'ab_GE',
			'ae' => 'ae_IR',
			'af' => 'af_ZA',
			'ak' => 'ak_GH',
			'am' => 'am_ET',
			'an' => 'an_ES',
			'ar' => 'ar_SA',
			'as' => 'as_IN',
			'av' => 'av_RU',
			'ay' => 'ay_BO',
			'az' => 'az_AZ',
			'ba' => 'ba_RU',
			'be' => 'be_BY',
			'bg' => 'bg_BG',
			'bh' => 'bh_IN',
			'bi' => 'bi_VU',
			'bm' => 'bm_ML',
			'bn' => 'bn_BD',
			'bo' => 'bo_CN',
			'br' => 'br_FR',
			'bs' => 'bs_BA',
			'ca' => 'ca_ES',
			'ce' => 'ce_RU',
			'ch' => 'ch_GU',
			'co' => 'co_FR',
			'cr' => 'cr_CA',
			'cs' => 'cs_CZ',
			'cu' => 'cu_RU',
			'cv' => 'cv_RU',
			'cy' => 'cy_GB',
			'da' => 'da_DK',
			'de' => 'de_DE',
			'dv' => 'dv_MV',
			'dz' => 'dz_BT',
			'ee' => 'ee_GH',
			'el' => 'el_GR',
			'en' => 'en_US',
			'en-gb' => 'en_GB',
			'eo' => 'eo_001',
			'es' => 'es_ES',
			'et' => 'et_EE',
			'eu' => 'eu_ES',
			'fa' => 'fa_IR',
			'ff' => 'ff_SN',
			'fi' => 'fi_FI',
			'fj' => 'fj_FJ',
			'fo' => 'fo_FO',
			'fr' => 'fr_FR',
			'fy' => 'fy_NL',
			'ga' => 'ga_IE',
			'gd' => 'gd_GB',
			'gl' => 'gl_ES',
			'gn' => 'gn_PY',
			'gu' => 'gu_IN',
			'gv' => 'gv_IM',
			'ha' => 'ha_NG',
			'he' => 'he_IL',
			'hi' => 'hi_IN',
			'ho' => 'ho_PG',
			'hr' => 'hr_HR',
			'ht' => 'ht_HT',
			'hu' => 'hu_HU',
			'hy' => 'hy_AM',
			'hz' => 'hz_NA',
			'ia' => 'ia_001',
			'id' => 'id_ID',
			'ie' => 'ie_001',
			'ig' => 'ig_NG',
			'ii' => 'ii_CN',
			'ik' => 'ik_US',
			'io' => 'io_001',
			'is' => 'is_IS',
			'it' => 'it_IT',
			'iu' => 'iu_CA',
			'ja' => 'ja_JP',
			'jv' => 'jv_ID',
			'ka' => 'ka_GE',
			'kg' => 'kg_CD',
			'ki' => 'ki_KE',
			'kj' => 'kj_NA',
			'kk' => 'kk_KZ',
			'kl' => 'kl_GL',
			'km' => 'km_KH',
			'kn' => 'kn_IN',
			'ko' => 'ko_KR',
			'kr' => 'kr_NG',
			'ks' => 'ks_IN',
			'ku' => 'ku_TR',
			'kv' => 'kv_RU',
			'kw' => 'kw_GB',
			'ky' => 'ky_KG',
			'la' => 'la_VA',
			'lb' => 'lb_LU',
			'lg' => 'lg_UG',
			'li' => 'li_NL',
			'ln' => 'ln_CD',
			'lo' => 'lo_LA',
			'lt' => 'lt_LT',
			'lu' => 'lu_CD',
			'lv' => 'lv_LV',
			'mg' => 'mg_MG',
			'mh' => 'mh_MH',
			'mi' => 'mi_NZ',
			'mk' => 'mk_MK',
			'ml' => 'ml_IN',
			'mn' => 'mn_MN',
			'mr' => 'mr_IN',
			'ms' => 'ms_MY',
			'mt' => 'mt_MT',
			'my' => 'my_MM',
			'na' => 'na_NR',
			'nb' => 'nb_NO',
			'nd' => 'nd_ZW',
			'ne' => 'ne_NP',
			'ng' => 'ng_NA',
			'nl' => 'nl_NL',
			'nn' => 'nn_NO',
			'no' => 'no_NO',
			'nr' => 'nr_ZA',
			'nv' => 'nv_US',
			'ny' => 'ny_MW',
			'oc' => 'oc_FR',
			'oj' => 'oj_CA',
			'om' => 'om_ET',
			'or' => 'or_IN',
			'os' => 'os_RU',
			'pa' => 'pa_IN',
			'pi' => 'pi_IN',
			'pl' => 'pl_PL',
			'ps' => 'ps_AF',
			'pt' => 'pt_PT',
			'pt-br' => 'pt_BR',
			'qu' => 'qu_PE',
			'rm' => 'rm_CH',
			'rn' => 'rn_BI',
			'ro' => 'ro_RO',
			'ru' => 'ru_RU',
			'rw' => 'rw_RW',
			'sa' => 'sa_IN',
			'sc' => 'sc_IT',
			'sd' => 'sd_PK',
			'se' => 'se_NO',
			'sg' => 'sg_CF',
			'si' => 'si_LK',
			'sk' => 'sk_SK',
			'sl' => 'sl_SI',
			'sm' => 'sm_WS',
			'sn' => 'sn_ZW',
			'so' => 'so_SO',
			'sq' => 'sq_AL',
			'sr' => 'sr_RS',
			'ss' => 'ss_ZA',
			'st' => 'st_LS',
			'su' => 'su_ID',
			'sv' => 'sv_SE',
			'sw' => 'sw_KE',
			'ta' => 'ta_IN',
			'te' => 'te_IN',
			'tg' => 'tg_TJ',
			'th' => 'th_TH',
			'ti' => 'ti_ET',
			'tk' => 'tk_TM',
			'tl' => 'tl_PH',
			'tn' => 'tn_BW',
			'to' => 'to_TO',
			'tr' => 'tr_TR',
			'ts' => 'ts_ZA',
			'tt' => 'tt_RU',
			'tw' => 'tw_GH',
			'ty' => 'ty_PF',
			'ug' => 'ug_CN',
			'uk' => 'uk_UA',
			'ur' => 'ur_PK',
			'uz' => 'uz_UZ',
			've' => 've_ZA',
			'vi' => 'vi_VN',
			'vo' => 'vo_001',
			'wa' => 'wa_BE',
			'wo' => 'wo_SN',
			'xh' => 'xh_ZA',
			'yi' => 'yi_001',
			'yo' => 'yo_NG',
			'za' => 'za_CN',
			'zh' => 'zh_CN',
			'zh-hk' => 'zh_HK',
			'zh-tw' => 'zh_TW',
			'zu' => 'zu_ZA'
		];

		$normalizedCode = strtolower(str_replace('_', '-', $code));

		return $map[$normalizedCode] ?? $code;
	}

	/**
	 * @return array<string, array<string, string>>
	 */
	public static function countriesByContinent(): array
	{
		return [
			__("Africa", Constants::TEXT_DOMAIN) => [
				"DZ" => __("Algeria", Constants::TEXT_DOMAIN),
				"AO" => __("Angola", Constants::TEXT_DOMAIN),
				"BJ" => __("Benin", Constants::TEXT_DOMAIN),
				"BW" => __("Botswana", Constants::TEXT_DOMAIN),
				"BF" => __("Burkina Faso", Constants::TEXT_DOMAIN),
				"BI" => __("Burundi", Constants::TEXT_DOMAIN),
				"CM" => __("Cameroon", Constants::TEXT_DOMAIN),
				"CV" => __("Cape Verde", Constants::TEXT_DOMAIN),
				"CF" => __("Central African Republic", Constants::TEXT_DOMAIN),
				"TD" => __("Chad", Constants::TEXT_DOMAIN),
				"KM" => __("Comoros", Constants::TEXT_DOMAIN),
				"CG" => __("Congo - Brazzaville", Constants::TEXT_DOMAIN),
				"CD" => __("Congo - Kinshasa", Constants::TEXT_DOMAIN),
				"DJ" => __("Djibouti", Constants::TEXT_DOMAIN),
				"EG" => __("Egypt", Constants::TEXT_DOMAIN),
				"CI" => __("Ivory Coast", Constants::TEXT_DOMAIN),
				"GQ" => __("Equatorial Guinea", Constants::TEXT_DOMAIN),
				"ER" => __("Eritrea", Constants::TEXT_DOMAIN),
				"ET" => __("Ethiopia", Constants::TEXT_DOMAIN),
				"GA" => __("Gabon", Constants::TEXT_DOMAIN),
				"GM" => __("Gambia", Constants::TEXT_DOMAIN),
				"GH" => __("Ghana", Constants::TEXT_DOMAIN),
				"GN" => __("Guinea", Constants::TEXT_DOMAIN),
				"GW" => __("Guinea-Bissau", Constants::TEXT_DOMAIN),
				"KE" => __("Kenya", Constants::TEXT_DOMAIN),
				"LS" => __("Lesotho", Constants::TEXT_DOMAIN),
				"LR" => __("Liberia", Constants::TEXT_DOMAIN),
				"LY" => __("Libya", Constants::TEXT_DOMAIN),
				"MG" => __("Madagascar", Constants::TEXT_DOMAIN),
				"MW" => __("Malawi", Constants::TEXT_DOMAIN),
				"ML" => __("Mali", Constants::TEXT_DOMAIN),
				"MA" => __("Morocco", Constants::TEXT_DOMAIN),
				"MR" => __("Mauritania", Constants::TEXT_DOMAIN),
				"MU" => __("Mauritius", Constants::TEXT_DOMAIN),
				"MZ" => __("Mozambique", Constants::TEXT_DOMAIN),
				"NA" => __("Namibia", Constants::TEXT_DOMAIN),
				"NE" => __("Niger", Constants::TEXT_DOMAIN),
				"NG" => __("Nigeria", Constants::TEXT_DOMAIN),
				"RW" => __("Rwanda", Constants::TEXT_DOMAIN),
				"ST" => __("São Tomé and Príncipe", Constants::TEXT_DOMAIN),
				"SN" => __("Senegal", Constants::TEXT_DOMAIN),
				"SC" => __("Seychelles", Constants::TEXT_DOMAIN),
				"SL" => __("Sierra Leone", Constants::TEXT_DOMAIN),
				"SO" => __("Somalia", Constants::TEXT_DOMAIN),
				"SD" => __("Sudan", Constants::TEXT_DOMAIN),
				"ZA" => __("South Africa", Constants::TEXT_DOMAIN),
				"SS" => __("South Sudan", Constants::TEXT_DOMAIN),
				"SZ" => __("Swaziland", Constants::TEXT_DOMAIN),
				"TZ" => __("Tanzania", Constants::TEXT_DOMAIN),
				"TG" => __("Togo", Constants::TEXT_DOMAIN),
				"TN" => __("Tunisia", Constants::TEXT_DOMAIN),
				"UG" => __("Uganda", Constants::TEXT_DOMAIN),
				"ZM" => __("Zambia", Constants::TEXT_DOMAIN),
				"ZW" => __("Zimbabwe", Constants::TEXT_DOMAIN)
			],
			__("Asia", Constants::TEXT_DOMAIN) => [
				"AF" => __("Afghanistan", Constants::TEXT_DOMAIN),
				"AM" => __("Armenia", Constants::TEXT_DOMAIN),
				"AZ" => __("Azerbaijan", Constants::TEXT_DOMAIN),
				"BH" => __("Bahrain", Constants::TEXT_DOMAIN),
				"BD" => __("Bangladesh", Constants::TEXT_DOMAIN),
				"BT" => __("Bhutan", Constants::TEXT_DOMAIN),
				"BN" => __("Brunei", Constants::TEXT_DOMAIN),
				"MM" => __("Burma", Constants::TEXT_DOMAIN),
				"KH" => __("Cambodia", Constants::TEXT_DOMAIN),
				"AE" => __("United Arab Emirates", Constants::TEXT_DOMAIN),
				"PH" => __("Philippines", Constants::TEXT_DOMAIN),
				"GE" => __("Georgia", Constants::TEXT_DOMAIN),
				"IN" => __("India", Constants::TEXT_DOMAIN),
				"ID" => __("Indonesia", Constants::TEXT_DOMAIN),
				"IQ" => __("Iraq", Constants::TEXT_DOMAIN),
				"IR" => __("Iran", Constants::TEXT_DOMAIN),
				"IL" => __("Israel", Constants::TEXT_DOMAIN),
				"JP" => __("Japan", Constants::TEXT_DOMAIN),
				"JO" => __("Jordan", Constants::TEXT_DOMAIN),
				"KZ" => __("Kazakhstan", Constants::TEXT_DOMAIN),
				"CN" => __("China", Constants::TEXT_DOMAIN),
				"KG" => __("Kyrgyzstan", Constants::TEXT_DOMAIN),
				"KW" => __("Kuwait", Constants::TEXT_DOMAIN),
				"LA" => __("Laos", Constants::TEXT_DOMAIN),
				"LB" => __("Lebanon", Constants::TEXT_DOMAIN),
				"MY" => __("Malaysia", Constants::TEXT_DOMAIN),
				"MV" => __("Maldives", Constants::TEXT_DOMAIN),
				"MN" => __("Mongolia", Constants::TEXT_DOMAIN),
				"NP" => __("Nepal", Constants::TEXT_DOMAIN),
				"KP" => __("North Korea", Constants::TEXT_DOMAIN),
				"OM" => __("Oman", Constants::TEXT_DOMAIN),
				"PK" => __("Pakistan", Constants::TEXT_DOMAIN),
				"PS" => __("Palestine", Constants::TEXT_DOMAIN),
				"QA" => __("Qatar", Constants::TEXT_DOMAIN),
				"RU" => __("Russia", Constants::TEXT_DOMAIN),
				"SA" => __("Saudi Arabia", Constants::TEXT_DOMAIN),
				"SG" => __("Singapore", Constants::TEXT_DOMAIN),
				"LK" => __("Sri Lanka", Constants::TEXT_DOMAIN),
				"KR" => __("South Korea", Constants::TEXT_DOMAIN),
				"SY" => __("Syria", Constants::TEXT_DOMAIN),
				"TJ" => __("Tajikistan", Constants::TEXT_DOMAIN),
				"TW" => __("Taiwan", Constants::TEXT_DOMAIN),
				"TH" => __("Thailand", Constants::TEXT_DOMAIN),
				"TL" => __("Timor-Leste", Constants::TEXT_DOMAIN),
				"TM" => __("Turkmenistan", Constants::TEXT_DOMAIN),
				"TR" => __("Turkey", Constants::TEXT_DOMAIN),
				"UZ" => __("Uzbekistan", Constants::TEXT_DOMAIN),
				"VN" => __("Vietnam", Constants::TEXT_DOMAIN),
				"YE" => __("Yemen", Constants::TEXT_DOMAIN)
			],
			__("Europe", Constants::TEXT_DOMAIN) => [
				"AL" => __("Albania", Constants::TEXT_DOMAIN),
				"AD" => __("Andorra", Constants::TEXT_DOMAIN),
				"BE" => __("Belgium", Constants::TEXT_DOMAIN),
				"BA" => __("Bosnia and Herzegovina", Constants::TEXT_DOMAIN),
				"BG" => __("Bulgaria", Constants::TEXT_DOMAIN),
				"CY" => __("Cyprus", Constants::TEXT_DOMAIN),
				"DK" => __("Denmark", Constants::TEXT_DOMAIN),
				"EE" => __("Estonia", Constants::TEXT_DOMAIN),
				"FI" => __("Finland", Constants::TEXT_DOMAIN),
				"FR" => __("France", Constants::TEXT_DOMAIN),
				"GR" => __("Greece", Constants::TEXT_DOMAIN),
				"NL" => __("Netherlands", Constants::TEXT_DOMAIN),
				"BY" => __("Belarus", Constants::TEXT_DOMAIN),
				"IE" => __("Ireland", Constants::TEXT_DOMAIN),
				"IS" => __("Iceland", Constants::TEXT_DOMAIN),
				"IT" => __("Italy", Constants::TEXT_DOMAIN),
				"XK" => __("Kosovo", Constants::TEXT_DOMAIN),
				"HR" => __("Croatia", Constants::TEXT_DOMAIN),
				"LV" => __("Latvia", Constants::TEXT_DOMAIN),
				"LI" => __("Liechtenstein", Constants::TEXT_DOMAIN),
				"LT" => __("Lithuania", Constants::TEXT_DOMAIN),
				"LU" => __("Luxembourg", Constants::TEXT_DOMAIN),
				"MT" => __("Malta", Constants::TEXT_DOMAIN),
				"MD" => __("Moldova", Constants::TEXT_DOMAIN),
				"MC" => __("Monaco", Constants::TEXT_DOMAIN),
				"ME" => __("Montenegro", Constants::TEXT_DOMAIN),
				"MK" => __("North Macedonia", Constants::TEXT_DOMAIN),
				"NO" => __("Norway", Constants::TEXT_DOMAIN),
				"PL" => __("Poland", Constants::TEXT_DOMAIN),
				"PT" => __("Portugal", Constants::TEXT_DOMAIN),
				"RO" => __("Romania", Constants::TEXT_DOMAIN),
				"RU" => __("Russia", Constants::TEXT_DOMAIN),
				"SM" => __("San Marino", Constants::TEXT_DOMAIN),
				"CH" => __("Switzerland", Constants::TEXT_DOMAIN),
				"RS" => __("Serbia", Constants::TEXT_DOMAIN),
				"SK" => __("Slovakia", Constants::TEXT_DOMAIN),
				"SI" => __("Slovenia", Constants::TEXT_DOMAIN),
				"ES" => __("Spain", Constants::TEXT_DOMAIN),
				"GB" => __("United Kingdom", Constants::TEXT_DOMAIN),
				"SE" => __("Sweden", Constants::TEXT_DOMAIN),
				"CZ" => __("Czech Republic", Constants::TEXT_DOMAIN),
				"DE" => __("Germany", Constants::TEXT_DOMAIN),
				"UA" => __("Ukraine", Constants::TEXT_DOMAIN),
				"HU" => __("Hungary", Constants::TEXT_DOMAIN),
				"VA" => __("Vatican City", Constants::TEXT_DOMAIN),
				"AT" => __("Austria", Constants::TEXT_DOMAIN)
			],
			__("North America", Constants::TEXT_DOMAIN) => [
				"AG" => __("Antigua and Barbuda", Constants::TEXT_DOMAIN),
				"BS" => __("Bahamas", Constants::TEXT_DOMAIN),
				"BB" => __("Barbados", Constants::TEXT_DOMAIN),
				"BZ" => __("Belize", Constants::TEXT_DOMAIN),
				"CA" => __("Canada", Constants::TEXT_DOMAIN),
				"CR" => __("Costa Rica", Constants::TEXT_DOMAIN),
				"CU" => __("Cuba", Constants::TEXT_DOMAIN),
				"DM" => __("Dominica", Constants::TEXT_DOMAIN),
				"DO" => __("Dominican Republic", Constants::TEXT_DOMAIN),
				"SV" => __("El Salvador", Constants::TEXT_DOMAIN),
				"GD" => __("Grenada", Constants::TEXT_DOMAIN),
				"GT" => __("Guatemala", Constants::TEXT_DOMAIN),
				"HT" => __("Haiti", Constants::TEXT_DOMAIN),
				"HN" => __("Honduras", Constants::TEXT_DOMAIN),
				"JM" => __("Jamaica", Constants::TEXT_DOMAIN),
				"MX" => __("Mexico", Constants::TEXT_DOMAIN),
				"NI" => __("Nicaragua", Constants::TEXT_DOMAIN),
				"PA" => __("Panama", Constants::TEXT_DOMAIN),
				"KN" => __("Saint Kitts and Nevis", Constants::TEXT_DOMAIN),
				"LC" => __("Saint Lucia", Constants::TEXT_DOMAIN),
				"VC" => __("Saint Vincent and the Grenadines", Constants::TEXT_DOMAIN),
				"TT" => __("Trinidad and Tobago", Constants::TEXT_DOMAIN),
				"US" => __("United States", Constants::TEXT_DOMAIN)
			],
			__("South America", Constants::TEXT_DOMAIN) => [
				"AR" => __("Argentina", Constants::TEXT_DOMAIN),
				"BO" => __("Bolivia", Constants::TEXT_DOMAIN),
				"BR" => __("Brazil", Constants::TEXT_DOMAIN),
				"CL" => __("Chile", Constants::TEXT_DOMAIN),
				"CO" => __("Colombia", Constants::TEXT_DOMAIN),
				"EC" => __("Ecuador", Constants::TEXT_DOMAIN),
				"GY" => __("Guyana", Constants::TEXT_DOMAIN),
				"PY" => __("Paraguay", Constants::TEXT_DOMAIN),
				"PE" => __("Peru", Constants::TEXT_DOMAIN),
				"SR" => __("Suriname", Constants::TEXT_DOMAIN),
				"UY" => __("Uruguay", Constants::TEXT_DOMAIN),
				"VE" => __("Venezuela", Constants::TEXT_DOMAIN)
			],
			__("Oceania", Constants::TEXT_DOMAIN) => [
				"AU" => __("Australia", Constants::TEXT_DOMAIN),
				"FJ" => __("Fiji", Constants::TEXT_DOMAIN),
				"KI" => __("Kiribati", Constants::TEXT_DOMAIN),
				"MH" => __("Marshall Islands", Constants::TEXT_DOMAIN),
				"FM" => __("Micronesia", Constants::TEXT_DOMAIN),
				"NR" => __("Nauru", Constants::TEXT_DOMAIN),
				"NZ" => __("New Zealand", Constants::TEXT_DOMAIN),
				"PW" => __("Palau", Constants::TEXT_DOMAIN),
				"PG" => __("Papua New Guinea", Constants::TEXT_DOMAIN),
				"WS" => __("Samoa", Constants::TEXT_DOMAIN),
				"SB" => __("Solomon Islands", Constants::TEXT_DOMAIN),
				"TO" => __("Tonga", Constants::TEXT_DOMAIN),
				"TV" => __("Tuvalu", Constants::TEXT_DOMAIN),
				"VU" => __("Vanuatu", Constants::TEXT_DOMAIN)
			]
		];
	}

	public static function getCountryName(string $code): string
	{
		$countries = self::countriesByContinent();
		foreach ($countries as $continent) {
			foreach ($continent as $countryCode => $countryName) {
				if ($countryCode === $code) {
					return $countryName;
				}
			}
		}

		return $code;
	}

	public static function removeCountryCode(string $langCountryCode): string
	{
		if (str_contains($langCountryCode, '_')) {
			[$langCode] = explode('_', $langCountryCode);
			return $langCode;
		}

		return $langCountryCode;
	}

	public static function getCurrentLanguage(): mixed
	{
		// WPML
		if (defined('ICL_LANGUAGE_CODE')) {
			return ICL_LANGUAGE_CODE;
		}

		if (function_exists('apply_filters')) {
			$lang = apply_filters('wpml_current_language', null);
			if (!empty($lang)) {
				return $lang;
			}
		}

		// Polylang
		if (function_exists('pll_current_language')) {
			return pll_current_language();
		}

		// TranslatePress
		if (function_exists('trp_get_current_language')) {
			return trp_get_current_language();
		}

		// Weglot
		if (defined('WEGLOT_CURRENT_LANG')) {
			return WEGLOT_CURRENT_LANG;
		}

		if (class_exists('\Weglot\Client\Context\Context')) {
			$context = '\Weglot\Client\Context\Context';
			if (method_exists($context, 'getCurrentLanguage')) { // @phpstan-ignore function.impossibleType
				return $context::getCurrentLanguage();
			}
		}

		// MultilingualPress
		if (function_exists('mlp_get_current_language')) {
			return mlp_get_current_language();
		}

		return get_locale();
	}

	public static function getDefaultLanguage(): mixed
	{
		// WPML
		if (function_exists('apply_filters')) {
			$lang = apply_filters('wpml_default_language', null);
			if (!empty($lang)) {
				return $lang;
			}
		}

		// Polylang
		if (function_exists('pll_default_language')) {
			return pll_default_language();
		}

		// TranslatePress
		if (function_exists('trp_get_default_language')) {
			return trp_get_default_language();
		}

		// Weglot
		if (defined('WEGLOT_LANGUAGE_FROM')) {
			return constant('WEGLOT_LANGUAGE_FROM');
		}

		// MultilingualPress
		if (function_exists('mlp_get_default_language')) {
			return mlp_get_default_language();
		}

		return get_site_option('WPLANG') ?: 'en_US';
	}

	/**
     * @return mixed[]
     */
    public static function getAlternateLocales(): array
	{
		$currentLocale = get_locale();
		$alternates = [];

		if (function_exists('icl_get_languages')) {
			$languages = icl_get_languages('skip_missing=0');
			if (!empty($languages)) {
				foreach ($languages as $lang) {
					$locale = str_replace('-', '_', $lang['default_locale']);
					if ($locale !== $currentLocale) {
						$alternates[] = $locale;
					}
				}
			}
		} elseif (function_exists('pll_get_languages')) {
			$languages = pll_get_languages(['raw' => 1]);
			if (!empty($languages)) {
				foreach ($languages as $lang) {
					if (!empty($lang['locale'])) {
						$locale = str_replace('-', '_', $lang['locale']);
						if ($locale !== $currentLocale) {
							$alternates[] = $locale;
						}
					}
				}
			}
		} elseif (function_exists('trp_get_languages')) {
			$languages = trp_get_languages();
			if (!empty($languages['language-names'])) {
				foreach ($languages['language-names'] as $code => $name) {
					$locale = LocaleHelper::getLocaleFromLanguageCode($code);
					if ($locale !== $currentLocale) {
						$alternates[] = $locale;
					}
				}
			}
		} elseif (class_exists('\WeglotWP\Services\Language')) {
			$languageService = new \WeglotWP\Services\Language();
			$languages = $languageService->get_languages();
			foreach ($languages as $lang) {
				$locale = str_replace('-', '_', $lang->getLocale());
				if ($locale !== $currentLocale) {
					$alternates[] = $locale;
				}
			}
		} elseif (function_exists('mlp_get_available_languages')) {
			$languages = mlp_get_available_languages();
			foreach ($languages as $lang) {
				$locale = LocaleHelper::getLocaleFromLanguageCode($lang);
				if ($locale !== $currentLocale) {
					$alternates[] = $locale;
				}
			}
		}

		return $alternates;
	}
}
