<?php

declare(strict_types=1);

namespace SignocoreToolkit\Support;

class GlossaryHelper
{
    /**
     * @return array<int, \WP_Post|null>
     */
    public static function termsByTitle(string $title): array
    {
        global $wpdb;

        $results = $wpdb->get_results($wpdb->prepare(sprintf("SELECT `ID` FROM `%s` WHERE `post_type` = 'wpsseo_term' AND `post_status` = 'publish' AND `post_title` LIKE %%s", $wpdb->posts), '%' . $wpdb->esc_like($title) . '%'));
        $return = [];

        if (!empty($results)) {
            foreach ($results as $term) {
                $return[] = get_post($term->ID);
            }
        }

        return $return;
    }

    /**
     * Batch version of termsByTitle() - single query for multiple titles.
     *
     * @param array<int, string> $titles
     * @return array<string, array<int, \WP_Post>> Keyed by title
     */
    public static function termsByTitles(array $titles): array
    {
        if ($titles === []) {
            return [];
        }

        global $wpdb;

        $likeClauses = [];
        $params = [];
        foreach ($titles as $title) {
            $likeClauses[] = '`post_title` LIKE %s';
            $params[] = '%' . $wpdb->esc_like($title) . '%';
        }

        $sql = sprintf(
            "SELECT `ID`, `post_title` FROM `%s` WHERE `post_type` = 'wpsseo_term' AND `post_status` = 'publish' AND (%s)",
            $wpdb->posts,
            implode(' OR ', $likeClauses)
        );

        $results = $wpdb->get_results($wpdb->prepare($sql, ...$params));
        $grouped = [];

        if (!empty($results)) {
            foreach ($results as $row) {
                $post = get_post($row->ID);
                if (!$post) {
                    continue;
                }
                foreach ($titles as $title) {
                    if (stripos($row->post_title, $title) !== false) {
                        $grouped[$title][] = $post;
                    }
                }
            }
        }

        return $grouped;
    }


    /**
     * @param array<int, string> $slugs
     * @param array<int, int> $exludeIds
     * @return array<int, \WP_Post>
     */
    public static function termsByTags(array $slugs, int $numberPosts = 10, array $exludeIds = []): array
    {
        $args = [
            'post_type' => 'wpsseo_term',
            'post_status' => 'publish',
            'meta_key' => 'page_views',
            'orderby' => 'meta_value_num',
            'order' => 'desc',
            'numberposts' => $numberPosts,
            'suppress_filters' => false,
            'no_found_rows'          => true,
            'ignore_sticky_posts'    => true,
            'update_post_meta_cache' => false,
            'tax_query' => [
                [
                    'taxonomy' => 'post_tag',
                    'field' => 'slug',
                    'terms' => $slugs,
                ]
            ]
        ];

        if ($exludeIds !== []) {
            $args['post__not_in'] = $exludeIds;
        }

        return get_posts($args);
    }

    /**
     * @param array<int, \WP_Post> $terms
     */
    public static function render(array $terms, string $title = '', string $class = 'glossary-terms'): string
    {
        if ($terms === []) {
            return '';
        }

        $output = '<div class="' . esc_attr($class) . '">';
        if ($title !== '' && $title !== '0') {
            $output .= '<h3>' . esc_html($title) . '</h3>';
        }

        $output .= '<ul>';

        foreach ($terms as $term) {
            $output .= '<li><a href="' . esc_url((string) get_permalink($term->ID)) . '">' . esc_html($term->post_title) . '</a></li>';
        }

        return $output . '</ul></div>';
    }
}
