<?php

declare(strict_types=1);

namespace SignocoreToolkit\Infrastructure\Traits;

use FilesystemIterator;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use SignocoreToolkit\Application\Constants;
use SplFileInfo;
use ZipArchive;

/**
 * Shared zip download functionality for creating and streaming zip archives.
 *
 * @package SignocoreToolkit\Infrastructure\Traits
 * @since 2.1.0
 */
trait ZipDownload
{
	/**
	 * Create a zip archive and stream it to the browser.
	 *
	 * @param string $sourceDir Directory to archive.
	 * @param string $zipName   Filename for the download.
	 */
	protected function createAndStreamZip(string $sourceDir, string $zipName): void
	{
		$tempFile = tempnam(sys_get_temp_dir(), 'signocore_zip_');
		if ($tempFile === false) {
			wp_die(
				esc_html__('Failed to create temporary file.', Constants::TEXT_DOMAIN),
				esc_html__('Download Error', Constants::TEXT_DOMAIN),
				['response' => 500]
			);
		}

		$zip = new ZipArchive();
		if ($zip->open($tempFile, ZipArchive::OVERWRITE) !== true) {
			unlink($tempFile);
			wp_die(
				esc_html__('Failed to create zip archive.', Constants::TEXT_DOMAIN),
				esc_html__('Download Error', Constants::TEXT_DOMAIN),
				['response' => 500]
			);
		}

		$this->addFilesToZip($sourceDir, $zip, strlen(dirname($sourceDir)) + 1);
		$zip->close();

		header('Content-Type: application/zip');
		header('Content-Disposition: attachment; filename="' . $zipName . '"');
		header('Content-Length: ' . filesize($tempFile));
		header('Pragma: no-cache');
		header('Expires: 0');

		readfile($tempFile);
		unlink($tempFile);
		exit;
	}

	/**
	 * Recursively add files to a zip archive.
	 *
	 * @param string     $folder  Directory to add.
	 * @param ZipArchive $zip     Zip archive instance.
	 * @param int        $baseLen Length of base path to strip.
	 */
	protected function addFilesToZip(string $folder, ZipArchive $zip, int $baseLen): void
	{
		$files = new RecursiveIteratorIterator(
			new RecursiveDirectoryIterator($folder, FilesystemIterator::SKIP_DOTS),
			RecursiveIteratorIterator::LEAVES_ONLY
		);

		/** @var SplFileInfo $file */
		foreach ($files as $file) {
			if (!$file->isFile()) {
				continue;
			}

			$filePath = $file->getRealPath();
			if (false === $filePath) {
				continue;
			}

			$relativePath = substr($filePath, $baseLen);

			if ($this->shouldSkipFile($relativePath)) {
				continue;
			}

			$zip->addFile($filePath, $relativePath);
		}
	}

	/**
	 * Determine if a file should be excluded from the zip.
	 *
	 * @param string $relativePath Relative file path.
	 * @return bool True if file should be skipped.
	 */
	protected function shouldSkipFile(string $relativePath): bool
	{
		$parts = explode(DIRECTORY_SEPARATOR, $relativePath);
		foreach ($parts as $part) {
			if ($part !== '' && $part[0] === '.') {
				return true;
			}
		}

		$filename = basename($relativePath);

		// Skip rector.php in root (e.g., plugin-name/rector.php)
		if ($filename === 'rector.php') {
			return true;
		}

		// Skip phpstan files in root (e.g., plugin-name/phpstan.neon)
		if (count($parts) === 2 && str_starts_with($filename, 'phpstan')) {
			return true;
		}

		// Skip markdown files
		$extension = strtolower(pathinfo($relativePath, PATHINFO_EXTENSION));
		return $extension === 'md';
	}
}
