<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features;

use SignocoreToolkit\Infrastructure\Traits\Singleton;
use SignocoreToolkit\Application\Constants;
use SignocoreToolkit\Support\FAQHelper;
use SignocoreToolkit\Support\GlossaryHelper;
use SignocoreToolkit\Support\SocialShareHelper;

final class WooCommerce
{
    use Singleton;

    protected function init(): void
    {
        if (!Constants::$isWooCommerce) {
            return;
        }

        add_action('init', [$this, 'beforeLoop']);
        add_filter('woocommerce_product_description_heading', '__return_null');
        add_filter('woocommerce_product_additional_information_heading', '__return_null');
        add_filter('woocommerce_subcategory_count_html', '__return_false');
        add_filter('woocommerce_product_tabs', [$this, 'addTabs']);
        add_action('woocommerce_share', [$this, 'addSocialShareButtons'], 100);
        add_filter('woocommerce_email_headers', [$this, 'changeReplyTo'], 10, 1);
        add_filter('woocommerce_product_tabs', [$this, 'removeEmptyDescription'], 20, 1);
        add_filter('woocommerce_product_single_add_to_cart_text', [$this, 'addToCartText'], 10, 2);
        add_filter('woocommerce_product_add_to_cart_text', [$this, 'addToCartText'], 10, 2);
        add_filter('woocommerce_package_rates', [$this, 'hideShippingWhenFreeIsAvailable'], 100);

        if (!Constants::$isAdmin) {
            $showSafeCheckout = get_option(Constants::OPTION_PREFIX . 'show_safe_checkout', 'after_summary');
            if ($showSafeCheckout === 'before_quantity') {
                add_action('woocommerce_before_add_to_cart_quantity', [$this, 'addSafeCheckoutIcons'], 0);
            } elseif ($showSafeCheckout === 'after_summary') {
                add_action('woocommerce_share', [$this, 'addSafeCheckoutIcons'], 90);
            }
        }
    }

    public function beforeLoop(): void
    {
        remove_action('woocommerce_before_main_content', 'woocommerce_breadcrumb', 20);

        if (!(bool) get_option(Constants::OPTION_PREFIX . 'show_ordering', true)) {
            remove_action('woocommerce_before_shop_loop', 'woocommerce_catalog_ordering', 30);
        }

        if (!(bool) get_option(Constants::OPTION_PREFIX . 'show_result_count', true)) {
            remove_action('woocommerce_before_shop_loop', 'woocommerce_result_count', 20);
        }
    }

    /**
     * @param array<string, array<string, mixed>> $tabs
     * @return array<string, array<string, mixed>>
     */
    public function addTabs(array $tabs): array
    {
        $showRelatedQuestions = (bool) get_option(Constants::OPTION_PREFIX . 'show_related_questions_products', true);
        if ($showRelatedQuestions && defined('SIGNOCORE_FAQ_VERSION')) {
            $relatedQuestions = $this->addRelatedQuestions();
            if ($relatedQuestions !== '' && $relatedQuestions !== '0') {
                $tabs['related_questions'] = [
                    'title' => __('Related Questions', Constants::TEXT_DOMAIN),
                    'priority' => 50,
                    'callback' => function () use ($relatedQuestions): void {
                        echo wp_kses_post($relatedQuestions);
                    }
                ];
            }
        }

        $showRelatedTerms = (bool) get_option(Constants::OPTION_PREFIX . 'show_related_glossary_terms_products', true);
        if ($showRelatedTerms && defined('SIGNOCORE_GLOSSARY_VERSION')) {
            $relatedTerms = $this->addRelatedTerms();
            if ($relatedTerms !== '' && $relatedTerms !== '0') {
                $tabs['related_terms'] = [
                    'title' => __('Related Terms', Constants::TEXT_DOMAIN),
                    'priority' => 60,
                    'callback' => function () use ($relatedTerms): void {
                        echo wp_kses_post($relatedTerms);
                    }
                ];
            }
        }

        return $tabs;
    }

    public function addRelatedQuestions(): string
    {
        $post = get_post();
        $relatedQuestions = get_transient('related_questions_' . $post->ID);

        if ($relatedQuestions === false) {
            $tags = wp_get_post_tags($post->ID, ['taxonomy' => 'product_tag', 'fields' => 'slugs']);

            if (empty($tags) || $tags instanceof \WP_Error) {
                return '';
            }

            /** @var array<int, string> $tags */
            $faqs = FAQHelper::faqsByTags($tags, 10);
            $relatedQuestions = '';

            if ($faqs !== []) {
                $relatedQuestions = FAQHelper::render($faqs, sprintf(__('Questions related to %s', Constants::TEXT_DOMAIN), $post->post_title), 'related-questions');

                set_transient('related_questions_' . $post->ID, $relatedQuestions, MONTH_IN_SECONDS);
            }
        }

        return is_string($relatedQuestions) ? $relatedQuestions : '';
    }

    public function addRelatedTerms(): string
    {
        $post = get_post();
        $relatedTerms = get_transient('related_terms_' . $post->ID);

        if ($relatedTerms === false) {
            $tags = get_the_terms($post->ID, 'product_tag');

            if (empty($tags) || $tags instanceof \WP_Error) {
                return '';
            }

            $tagSlugs = [];
            $tagNames = [];

            foreach ($tags as $tag) {
                $tagSlugs[] = $tag->slug;
                $tagNames[] = $tag->name;
            }

            $termsByTitle = [];
            $excludeIds = [];
            $batchResults = GlossaryHelper::termsByTitles($tagNames);
            foreach ($batchResults as $terms) {
                foreach ($terms as $term) {
                    if (!isset($excludeIds[$term->ID])) {
                        $termsByTitle[] = $term;
                        $excludeIds[$term->ID] = true;
                    }
                }
            }

            $termsByTags = GlossaryHelper::termsByTags($tagSlugs, 10, array_keys($excludeIds));
            $allTerms = array_merge($termsByTitle, $termsByTags);

            if ($allTerms !== []) {
                update_postmeta_cache(array_column($allTerms, 'ID'));
            }

            usort($allTerms, function ($a, $b): int {
                $aViews = get_post_meta($a->ID, 'page_views', true) ?: 0;
                $bViews = get_post_meta($b->ID, 'page_views', true) ?: 0;
                return $bViews <=> $aViews;
            });

            $allTerms = array_slice($allTerms, 0, 10);
            $relatedTerms = '';

            if ($allTerms !== []) {
                $relatedTerms = GlossaryHelper::render($allTerms, sprintf(__('Terms related to %s', Constants::TEXT_DOMAIN), $post->post_title), 'related-words');

                set_transient('related_terms_' . $post->ID, $relatedTerms, MONTH_IN_SECONDS);
            }
        }

        return is_string($relatedTerms) ? $relatedTerms : '';
    }

    /**
     * Add social share buttons to WooCommerce product pages.
     *
     * Uses SocialShareHelper for consistent share button generation.
     */
    public function addSocialShareButtons(): void
    {
        if (!is_singular('product')) {
            return;
        }

        $postTypes = array_filter(array_merge(array_keys(get_option(Constants::OPTION_PREFIX . 'social_share_locations', []))));
        if (!in_array('product', $postTypes)) {
            return;
        }

        echo SocialShareHelper::generateShareButtons(null, 'product-social-share');
    }

    public function addSafeCheckoutIcons(): void
    {
        $safeCheckoutIcons = get_transient('safe_checkout_icons');

        if ($safeCheckoutIcons === false) {
            $safeCheckoutIcons = '<fieldset class="safe-checkout-icons">
				<legend>' . __('Guaranteed Safe checkout', Constants::TEXT_DOMAIN) . '</legend>
				' . apply_shortcodes('[payment-methods]') . '
			</fieldset>';

            set_transient('safe_checkout_icons', $safeCheckoutIcons, MONTH_IN_SECONDS);
        }

        echo $safeCheckoutIcons;
    }

    public function changeReplyTo(string $header): string
    {
        $companyName = get_option('wpsseo_company_name', get_bloginfo('name'));
        $companyEmail = get_option('wpsseo_company_email', get_bloginfo('admin_email'));

        if (!empty($companyEmail)) {
            $header .= 'Reply-to: ' . $companyName . ' <' . $companyEmail . ">\r\n";
        }

        return $header;
    }

    /**
     * @param array<string, array<string, mixed>> $tabs
     * @return array<string, array<string, mixed>>
     */
    public function removeEmptyDescription(array $tabs): array
    {
        $post = get_post();

        if (empty($post->post_content)) {
            unset($tabs['description']);
        }

        return $tabs;
    }

    public function addToCartText(string $text, object $product): string
    {
        if (Constants::$isAdmin) {
            return $text;
        }

        if (!is_singular('product') && method_exists($product, 'get_type') && $product->get_type() === 'variable') {
            return __('View product', Constants::TEXT_DOMAIN);
        }

        return $text;
    }

    /**
     * @param array<string, object> $rates
     * @return array<string, object>
     */
    public function hideShippingWhenFreeIsAvailable(array $rates): array
    {
        $newRates = [];
        $freeRates = [];
        $localRates = [];

        foreach ($rates as $rateId => $rate) {
            /** @var object{method_id: string} $rate */
            if ('free_shipping' === $rate->method_id) {
                $freeRates[$rateId] = $rate;
            }

            if ('local_pickup' === $rate->method_id) {
                $localRates[$rateId] = $rate;
            }
        }

        if ($freeRates !== []) {
            $newRates = array_merge($freeRates, $localRates);
        }

        return $newRates === [] ? $rates : $newRates;
    }
}
