<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features;

use SignocoreToolkit\Infrastructure\Traits\Singleton;
use SignocoreToolkit\Application\Constants;
use SignocoreToolkit\Support\SocialShareHelper;

final class Social
{
    use Singleton;

    protected function init(): void
    {
        add_action('kadence_single_after_entry_content', [$this, 'addSocialShareButtons']);
        add_action('save_post', [$this, 'clearSocialTransients']);
    }

    public function registerHooks(): void
    {
        add_action('wp_head', [$this, 'addSocialShareButtons']);
    }

    private function isWooCommercePage(): bool
    {
        if (!Constants::$isWooCommerce) {
            return false;
        }

        return is_woocommerce() || is_product() || is_shop() || is_cart() || is_checkout() || is_account_page() || is_view_order_page() || is_checkout_pay_page() || is_edit_account_page() || is_order_received_page() || is_add_payment_method_page() || is_lost_password_page();
    }

    /**
     * Add social share buttons to content.
     *
     * Uses SocialShareHelper for consistent share button generation.
     */
    public function addSocialShareButtons(): void
    {
        if (is_home() || is_front_page() || $this->isWooCommercePage()) {
            return;
        }

        /** @var array<string, string> $locations */
        $locations = get_option('signocore_theme_social_share_locations') ?: get_option('swift_theme_social_share_locations', ['post' => __('Post')]);
        $postTypes = array_values(array_filter(array_keys($locations)));
        if ($postTypes !== [] && is_singular($postTypes)) {
            echo SocialShareHelper::generateShareButtons();
        }
    }

    /**
     * Clear social share cache when post is saved.
     *
     * @param int $postId The post ID
     */
    public function clearSocialTransients(int $postId): void
    {
        SocialShareHelper::clearCache($postId);
    }
}
