<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features;

use SignocoreToolkit\Infrastructure\Traits\Singleton;
use SignocoreToolkit\Application\Constants;
use SignocoreToolkit\Support\LocaleHelper;

final class Shortcodes
{
    use Singleton;

    /** @var array<string, mixed> */
    private array $options;

    private bool $countryFormatted = false;

    protected function init(): void
    {
        $this->options = get_option('wpsseo_schema_data', []);

        add_shortcode('payment-methods', [$this, 'paymentMethods']);
        add_shortcode('company-name', [$this, 'nameShortcode']);
        add_shortcode('phone', [$this, 'phoneShortcode']);
        add_shortcode('email', [$this, 'emailShortcode']);
        add_shortcode('address', [$this, 'addressShortcode']);
        add_shortcode('zip', [$this, 'zipShortcode']);
        add_shortcode('city', [$this, 'cityShortcode']);
        add_shortcode('country', [$this, 'countryShortcode']);
        add_shortcode('vat', [$this, 'vatShortcode']);
        add_shortcode('contact', [$this, 'contactShortcode']);
        add_shortcode('map', [$this, 'mapShortcode']);
        add_shortcode('social-links', [$this, 'socialLinksShortcode']);
    }

    private function formatCountry(): void
    {
        if ($this->countryFormatted) {
            return;
        }

        $this->countryFormatted = true;

        if (!empty($this->options['company_country'])) {
            $countryCode = $this->options['company_country'];
            $countryName = LocaleHelper::getCountryName($countryCode);

            $this->options['company_country'] = $countryName;
        }
    }

    public function paymentMethods(): string
    {
        if (!Constants::$isWooCommerce) {
            return '';
        }

        $chosenPaymentMethods = get_option('signocore_theme_payment_methods') ?: get_option('swift_theme_payment_methods', []);
        $methods = '';

        if (!empty($chosenPaymentMethods)) {
            $methods .= '<div class="payment-logos">';

            foreach (array_keys($chosenPaymentMethods) as $method) {
                $alt = esc_attr(ucfirst((string) $method) . ' ' . __('Logo', Constants::TEXT_DOMAIN));
                $methods .= '<img src="' . Constants::$assetsUri . '/img/payment-methods/' . $method . '.png" alt="' . $alt . '">';
            }

            $methods .= '</div>';
        }

        return $methods;
    }

    public function nameShortcode(): string
    {
        $companyName = empty($this->options['company_name']) ? get_option('blogname') : $this->options['company_name'];

        return '<span translate="no">' . $companyName . '</span>';
    }

    /**
     * @param array<string, mixed>|string $atts
     */
    public function phoneShortcode($atts): string
    {
        /** @var array<string, string|bool> $a */
        $a = shortcode_atts([
            'link' => true,
            'icon' => true,
        ], is_array($atts) ? $atts : []);

        if (empty($this->options['company_phone'])) {
            return '';
        }

        $phone = (string) $this->options['company_phone'];
        $tel = str_replace([' ', '(', ')', '-'], '', $phone);
        $link = $a['link'] === 'true' || $a['link'] === true;
        $icon = $a['icon'] === 'true' || $a['icon'] === true;

        if ($link && $tel !== '') {
            return '<a href="tel:' . $tel . '" class="phone">' . ($icon ? '<i class="si si-phone" aria-hidden="true"></i> ' : '') . $phone . '</a>';
        }

        return $phone;
    }

    /**
     * @param array<string, mixed>|string $atts
     */
    public function emailShortcode($atts): string
    {
        /** @var array<string, string|bool|null> $a */
        $a = shortcode_atts([
            'link' => true,
            'icon' => false,
            'text' => null,
            'subject' => null,
        ], is_array($atts) ? $atts : []);

        $email = empty($this->options['company_email']) ? get_option('admin_email') : $this->options['company_email'];
        $email = Mail::emailToEntities($email);

        $link = $a['link'] === 'true' || $a['link'] === true;
        $icon = $a['icon'] === 'true' || $a['icon'] === true;

        if ($link && ($email !== '' && $email !== '0')) {
            $emailText = empty($a['text']) ? $email : $a['text'];
            $emailSubject = empty($a['subject']) ? null : '?subject=' . $a['subject'];
            return '<a href="mailto:' . $email . $emailSubject . '" class="email">' . ($icon === false ? '' : '<i class="si si-mail" aria-hidden="true"></i> ') . $emailText . '</a>';
        }

        return $email;
    }

    public function addressShortcode(): string
    {
        $address = '';

        if (!empty($this->options['company_co_address'])) {
            $address .= $this->options['company_co_address'] . ', ';
        }

        if (!empty($this->options['company_address'])) {
            $address .= $this->options['company_address'];
        }

        if (!empty($this->options['company_address_2'])) {
            $address .= ', ' . $this->options['company_address_2'];
        }

        return $address;
    }

    public function zipShortcode(): string
    {
        if (empty($this->options['company_zip'])) {
            return '';
        }

        return (string) $this->options['company_zip'];
    }

    public function cityShortcode(): string
    {
        if (empty($this->options['company_city'])) {
            return '';
        }

        return (string) $this->options['company_city'];
    }

    public function countryShortcode(): string
    {
        $this->formatCountry();

        if (empty($this->options['company_country'])) {
            return '';
        }

        return (string) $this->options['company_country'];
    }

    public function vatShortcode(): string
    {
        if (empty($this->options['company_vat'])) {
            return '';
        }

        return (string) $this->options['company_vat'];
    }

    /**
     * @param array<string, mixed>|string $atts
     */
    public function contactShortcode($atts): string
    {
        /** @var array<string, string|bool> $a */
        $a = shortcode_atts([
            'icons' => true,
            'hide_company' => true,
            'hide_country' => true,
            'hide_vat' => false,
            'inline_address' => true,
            'highlight_company' => true,
        ], is_array($atts) ? $atts : []);

        $icons = $a['icons'] === 'false' ? 'false' : 'true';
        $hideCompany = $a['hide_company'] !== 'false';
        $hideCountry = $a['hide_country'] !== 'false';
        $hideVat = $a['hide_vat'] !== 'false';
        $inlineAddress = $a['inline_address'] !== 'false';
        $highlight = $a['highlight_company'] !== 'false';
        $separator = $inlineAddress ? ', ' : '<br>';

        $this->formatCountry();

        $phone = apply_shortcodes('[phone icon="' . $icons . '"]');
        $email = apply_shortcodes('[email icon="' . $icons . '"]');
        $companyName = empty($this->options['company_name']) ? get_option('blogname') : $this->options['company_name'];
        $contact = '<div class="contact" translate="no"><p>';

        if (!$hideCompany) {
            $contact .= $companyName;

            if ($highlight) {
                $contact .= sprintf('<strong>%s</strong>', $companyName);
            }

            $contact .= '<br>';
        }

        $address = empty($this->options['company_co_address']) ? '' : $separator . $this->options['company_co_address'];
        $address .= empty($this->options['company_address']) ? '' : $separator . $this->options['company_address'];
        $address .= empty($this->options['company_address_2']) ? '' : $separator . $this->options['company_address_2'];
        $address .= !empty($this->options['company_zip']) && !empty($this->options['company_city']) ? $separator . $this->options['company_zip'] . ' ' . $this->options['company_city'] : '';
        $address .= !$hideCountry && !empty($this->options['company_country']) ? $separator . $this->options['company_country'] : '';

        $contact .= trim($address, $separator);

        $contact .= empty($email) ? '' : '<br>' . $email;
        $contact .= empty($phone) ? '' : '<br>' . $phone;
        $contact .= !$hideVat && !empty($this->options['company_vat']) ? '<br>' . __('VAT: ', Constants::TEXT_DOMAIN) . '<a href="#" class="vat">' . $this->options['company_vat'] . '</a>' : '';

        return $contact . '</p></div>';
    }

    /**
     * @param array<string, mixed>|string $atts
     */
    public function mapShortcode($atts): string
    {
        $apiKey = get_option('signocore_theme_google_maps_key') ?: get_option('swift_theme_google_maps_key');
        if (empty($apiKey)) {
            if (current_user_can('manage_options')) {
                return __('Invalid Google Maps API key. This message is only shown for administrators.', Constants::TEXT_DOMAIN);
            }

            return '';
        }

        /** @var array<string, string|int|bool> $a */
        $a = shortcode_atts([
            'width' => 600,
            'height' => 450,
            'has_gmb' => true,
            'address' => '',
            'zoom' => 16,
            'class' => '',
        ], is_array($atts) ? $atts : []);

        $this->formatCountry();

        $address = (string) $a['address'];
        if ($address === '') {
            if ($a['has_gmb'] !== 'false' && !empty($this->options['company_name'])) {
                $address = (string) $this->options['company_name'];
            }

            $address .= empty($this->options['company_address']) ? '' : ', ' . $this->options['company_address'];
            $address .= !empty($this->options['company_zip']) && !empty($this->options['company_city']) ? ', ' . $this->options['company_zip'] . ' ' . $this->options['company_city'] : '';
            $address .= empty($this->options['company_country']) ? '' : ', ' . $this->options['company_country'];
        }

        if ($address === '') {
            return '';
        }

        $width = (int) $a['width'];
        $height = (int) $a['height'];
        $responsiveDimensions = round(($height / $width) * 100, 2);
        $class = (string) $a['class'];

        return '<div class="map-iframe responsive-iframe' . ($class === '' ? '' : ' ' . $class) . '" style="padding-bottom: ' . $responsiveDimensions . '%;"><iframe loading="lazy" width="' . $width . '" height="' . $height . '" style="border: 0;" src="https://www.google.com/maps/embed/v1/place?q=' . urlencode($address) . '&key=' . $apiKey . '&zoom=' . $a['zoom'] . '" allowfullscreen></iframe></div>';
    }

    public function socialLinksShortcode(): string
    {
        $socialProfiles = empty($this->options['social_profiles']) ? [] : array_filter((array) $this->options['social_profiles']);
        $socialLinks = '';

        if ($socialProfiles !== []) {
            $socialLinks .= '<div class="social-icons">';

            foreach ($socialProfiles as $key => $url) {
                $socialLinks .= '<a class="' . $key . '" href="' . $url . '" target="_blank" rel="external noopener" title="' . ucwords((string) $key) . '"><i class="si si-' . $key . '" aria-hidden="true"></i></a>';
            }

            $socialLinks .= '</div>';
        }

        return $socialLinks;
    }
}
