<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features;

use SignocoreToolkit\Infrastructure\Traits\Singleton;

final class Images
{
	use Singleton;

	/** @var array<string, bool> */
	private array $orientationFixed = [];

	/** @var array<string, array<string, mixed>> */
	private array $previousMeta = [];

	protected function init(): void
	{
		add_filter('wp_handle_upload_prefilter', [$this, 'filterWpHandleUploadPrefilter'], 10, 1);
		add_filter('wp_handle_upload', [$this, 'filterWpHandleUpload'], 1, 1);
		add_action('after_setup_theme', [$this, 'updateImageSizes'], 999);
		add_filter('intermediate_image_sizes_advanced', [$this, 'disableImageSizes']);
	}

	/**
     * @param array<string, mixed> $file
     * @return array<string, mixed>
     */
    public function filterWpHandleUpload(array $file): array
	{
		$suffix = substr($file['file'], strrpos($file['file'], '.', -1) + 1);
		if (in_array($suffix, ['jpg', 'jpeg', 'tiff'], true)) {
			$this->fixImageOrientation($file['file']);
		}

		return $file;
	}

	/**
     * @param array<string, mixed> $file
     * @return array<string, mixed>
     */
    public function filterWpHandleUploadPrefilter(array $file): array
	{
		$suffix = substr($file['name'], strrpos($file['name'], '.', -1) + 1);
		if (in_array($suffix, ['jpg', 'jpeg', 'tiff'], true)) {
			$this->fixImageOrientation($file['tmp_name']);
		}

		return $file;
	}

	/**
     * Fix image orientation based on EXIF data with error handling.
     *
     * @param string $file Path to the image file
     */
    public function fixImageOrientation(string $file): void
	{
		if (isset($this->orientationFixed[$file])) {
			return;
		}

		try {
			// Suppress warnings from exif_read_data as it can be noisy
			$exif = @exif_read_data($file);

			if (!$exif || !isset($exif['Orientation']) || $exif['Orientation'] <= 1) {
				return;
			}

			// Include required WordPress files
			if (!function_exists('wp_get_image_editor')) {
				include_once(ABSPATH . 'wp-admin/includes/image.php');
			}
            
			if (!function_exists('wp_read_image_metadata')) {
				include_once(ABSPATH . 'wp-admin/includes/image-edit.php');
			}

			$operations = $this->calculateFlipAndRotate($file, $exif);

			if ($operations !== false) {
				$success = $this->doFlipAndRotate($file, $operations);

				if (!$success && defined('WP_DEBUG') && WP_DEBUG) {
					error_log(sprintf(
						'Signocore Toolkit: Failed to fix orientation for image: %s',
						basename($file)
					));
				}
			}
		} catch (\Exception $exception) {
			// Log EXIF read errors for debugging
			if (defined('WP_DEBUG') && WP_DEBUG) {
				error_log(sprintf(
					'Signocore Toolkit: EXIF read error for %s: %s',
					basename($file),
					$exception->getMessage()
				));
			}
		}
	}

	/**
     * @param array<string, mixed> $exif
     * @return false|array<string, mixed>
     */
    private function calculateFlipAndRotate(string $file, array $exif): false|array
	{
		$rotator = false;
		$flipper = false;

		// Lets switch to the orientation defined in the exif data.
		switch ($exif['Orientation']) {
			case 1:
				// We don't want to fix an already correct image :).
				$this->orientationFixed[$file] = true;
				return false;
			case 2:
				$flipper = [false, true];
				break;
			case 3:
				$orientation = -180;
				$rotator = true;
				break;
			case 4:
				$flipper = [true, false];
				break;
			case 5:
				$orientation = -90;
				$rotator = true;
				$flipper = [false, true];
				break;
			case 6:
				$orientation = -90;
				$rotator = true;
				break;
			case 7:
				$orientation = -270;
				$rotator = true;
				$flipper = [false, true];
				break;
			case 8:
			case 9:
				$orientation = -270;
				$rotator = true;
				break;
			default:
				$orientation = 0;
				$rotator = true;
				break;
		}

		return ['orientation' => $orientation ?? 0, 'rotator' => $rotator, 'flipper' => $flipper];
	}

	/**
     * Perform flip and rotate operations on image with error handling.
     *
     * @param string $file Path to the image file
     * @param array<string, mixed> $operations Operations to perform (rotation, flip)
     * @return bool True on success, false on failure
     */
    private function doFlipAndRotate(string $file, array $operations): bool
	{
		$editor = wp_get_image_editor($file);

		if (is_wp_error($editor)) {
			if (defined('WP_DEBUG') && WP_DEBUG) {
				error_log(sprintf(
					'Signocore Toolkit: Image editor failed for %s: %s',
					basename($file),
					$editor->get_error_message()
				));
			}
            
			return false;
		}

		// Save metadata if using GD editor (it strips EXIF data)
		if ('WP_Image_Editor_GD' === $editor::class) {
			$savedMeta = wp_read_image_metadata($file);
			if (is_array($savedMeta)) {
				$this->previousMeta[$file] = $savedMeta;
			}
		}

		try {
			// Perform rotation
			if (true === $operations['rotator']) {
				$result = $editor->rotate($operations['orientation']);
				if (is_wp_error($result)) {
					throw new \Exception($result->get_error_message());
				}
			}

			// Perform flip
			if (false !== $operations['flipper']) {
				$result = $editor->flip($operations['flipper'][0], $operations['flipper'][1]);
				if (is_wp_error($result)) {
					throw new \Exception($result->get_error_message());
				}
			}

			// Save the image
			$result = $editor->save($file);
			if (is_wp_error($result)) {
				throw new \Exception($result->get_error_message());
			}

			$this->orientationFixed[$file] = true;
			add_filter('wp_read_image_metadata', [$this, 'restoreMetaData'], 10, 2);

			return true;
		} catch (\Exception $exception) {
			if (defined('WP_DEBUG') && WP_DEBUG) {
				error_log(sprintf(
					'Signocore Toolkit: Image manipulation failed for %s: %s',
					basename($file),
					$exception->getMessage()
				));
			}
            
			return false;
		}
	}

	/**
     * @param array<string, mixed> $meta
     * @return array<string, mixed>
     */
    public function restoreMetaData(array $meta, string $file): array
	{
		if (isset($this->previousMeta[$file])) {
			$meta = $this->previousMeta[$file];

			$meta['orientation'] = 1;

			return $meta;
		}

		return $meta;
	}

	/**
     * @return mixed[]
     */
    public function getImageSizes(): array
	{
		global $_wp_additional_image_sizes;

		$thumbCrop = (bool) get_option('thumbnail_crop', true);

		$sizes = [
			'thumbnail' => [
				'type'      => 'default',
				'width'     => get_option('thumbnail_size_w'),
				'height'    => get_option('thumbnail_size_h'),
				'cropped'   => $thumbCrop
			],
			'medium' => [
				'type'      => 'default',
				'width'     => get_option('medium_size_w'),
				'height'    => get_option('medium_size_h'),
				'cropped'   => $thumbCrop
			],
			'medium_large' => [
				'type'      => 'default',
				'width'     => get_option('medium_large_size_w'),
				'height'    => get_option('medium_large_size_h'),
				'cropped'   => $thumbCrop
			],
			'large' => [
				'type'      => 'default',
				'width'     => get_option('large_size_w'),
				'height'    => get_option('large_size_h'),
				'cropped'   => $thumbCrop
			],
		];

		if (is_array($_wp_additional_image_sizes) && $_wp_additional_image_sizes !== []) {
			foreach ($_wp_additional_image_sizes as $size => $data) {
				$sizes[$size] = [
					'type'      => 'custom',
					'width'     => $data['width'],
					'height'    => $data['height'],
					'cropped'   => (bool) ($data['crop'] ?? true)
				];
			}
		}

		return $sizes;
	}

	public function updateImageSizes(): void
	{
		$sizes = $this->getImageSizes();
		$hash = md5((string) wp_json_encode($sizes));

		if (get_option('_image-sizes-hash') === $hash) {
			return;
		}

		update_option('_image-sizes', $sizes);
		update_option('_image-sizes-hash', $hash, false);
	}

	/**
     * @param array<string, mixed> $sizes
     * @return array<string, mixed>
     */
    public function disableImageSizes(array $sizes): array
	{
		$disable = ['1536x1536', '2048x2048'];

		foreach ($disable as $size) {
			unset($sizes[$size]);
		}

		return $sizes;
	}
}
