<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features;

use SignocoreToolkit\Infrastructure\Traits\Singleton;
use SignocoreToolkit\Application\Constants;

final class Files
{
	use Singleton;

	/**
     * @var array<string, string[]>
     */
    private array $cacheConfig = [
		'30d' => ['/\.gif$/', '/\.jpg$/', '/\.jpeg$/', '/\.png$/', '/\.svg$/', '/\.ico$/', '/\.woff$/', '/\.woff2$/', '/\.ttf$/', '/\.otf$/', '/\.eot$/'],
		'7d'  => ['/\.css$/', '/\.js$/']
	];

	protected function init(): void
	{
		add_filter('sanitize_file_name', [$this, 'sanitizeFilename'], 10);
		add_filter('upload_mimes', [$this, 'allowFileTypes']);
		add_action('send_headers', [$this, 'setHeaders']);
	}

	public function sanitizeFilename(string $filename): string
	{
		$pathinfo = pathinfo($filename);

		if (empty($pathinfo['extension'])) {
			return $filename;
		}

		$name = sanitize_title($pathinfo['filename']);
		$ext = $pathinfo['extension'];

		if (strlen($name) > 59) {
			$name = substr($name, 0, 59);
		}

		return sprintf('%s.%s', $name, $ext);
	}

	/**
     * @param array<string, mixed> $fileTypes
     * @return array<string, mixed>
     */
    public function allowFileTypes(array $fileTypes): array
	{
		$fileTypes['svg'] = 'image/svg+xml';
		$fileTypes['svgz'] = 'image/svg+xml';

		return $fileTypes;
	}

	public function setHeaders(): void
	{
		if (!Constants::$isPublic || wp_get_environment_type() !== 'production' || Constants::$isAdmin || is_preview()) {
			return;
		}

		if (defined('REST_REQUEST') && REST_REQUEST) {
			header('Cache-Control: no-cache, must-revalidate');
			header('Expires: ' . gmdate('D, d M Y H:i:s', time() - 3600) . ' GMT');
			return;
		}

        $uri = $_SERVER['REQUEST_URI'];
        foreach ($this->cacheConfig as $duration => $fileTypes) {
				foreach ($fileTypes as $type) {
					if (preg_match($type, $uri)) {
						header('Cache-Control: public, max-age=' . $this->convertDurationToSeconds($duration));
						return;
					}
				}
			}

        $cacheMaxAge = 3600;
        $frontPageId = get_option('page_on_front');
        $blogPageId = get_option('page_for_posts');
        if (is_front_page() && $frontPageId) {
				$lastModified = get_post_modified_time('U', false, $frontPageId);
			} elseif (is_home() && $blogPageId) {
				$args = [
					'post_type' => 'post',
					'posts_per_page' => 1,
					'orderby' => 'modified',
					'order' => 'DESC',
					'no_found_rows' => true,
					'fields' => 'ids',
				];
				$recentPost = get_posts($args);
				if (!empty($recentPost)) {
					$lastModified = get_post_modified_time('U', false, $recentPost[0]);
				} else {
					$lastModified = get_post_modified_time('U', false, $blogPageId);
				}
			} elseif (is_singular()) {
				global $post;
				$lastModified = get_post_modified_time('U', false, $post->ID);
			} else {
				header('Cache-Control: public, max-age=' . $cacheMaxAge . ', must-revalidate');
				header('Expires: ' . gmdate('D, d M Y H:i:s', time() + $cacheMaxAge) . ' GMT');
				return;
			}

        header('Cache-Control: public, max-age=' . $cacheMaxAge . ', must-revalidate');
        header('Expires: ' . gmdate('D, d M Y H:i:s', time() + $cacheMaxAge) . ' GMT');
        header('Last-Modified: ' . gmdate('D, d M Y H:i:s', (int) $lastModified) . ' GMT');
	}

	private function convertDurationToSeconds(string $duration): int
	{
		$daysToSeconds = [
			'd' => 86400,
			'h' => 3600,
			'm' => 60,
		];

		$timeUnit = substr($duration, -1);
		$timeValue = (int)substr($duration, 0, -1);

		return $timeValue * $daysToSeconds[$timeUnit];
	}
}
