<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features\DevTools;

use SignocoreToolkit\Application\Constants;
use SignocoreToolkit\Infrastructure\Traits\Singleton;

/**
 * User switching functionality.
 *
 * @package SignocoreToolkit\Features\DevTools
 * @since 3.0.0
 */
final class UserSwitcher
{
	use Singleton;

	/**
	 * User meta key for storing the original user ID during impersonation.
	 */
	private const ORIGINAL_USER_META = 'sctk_original_user_id';

	/**
	 * Initialize user switcher features.
	 *
	 * Registers hooks for user list row actions, switch handling,
	 * and the floating switch-back bar.
	 */
	protected function init(): void
	{
		add_filter('user_row_actions', [$this, 'addSwitchLink'], 10, 2);
		add_action('admin_notices', [$this, 'renderProfileSwitchButton']);
		add_action('admin_init', [$this, 'handleSwitch']);
		add_action('admin_post_sctk_switch_back', [$this, 'handleSwitchBack']);
		add_action('wp_footer', [$this, 'renderSwitchBackBar']);
		add_action('admin_footer', [$this, 'renderSwitchBackBar']);
	}

	/**
	 * Add "Switch to" link in user list row actions.
	 *
	 * Adds a quick-switch link for each user in the WordPress user
	 * list table, guarded by capability and impersonation checks.
	 *
	 * @param array<string, string> $actions Existing row actions.
	 * @param \WP_User              $user    The user for this row.
	 * @return array<string, string> Modified row actions.
	 */
	public function addSwitchLink(array $actions, \WP_User $user): array
	{
		if (!current_user_can('manage_options')) {
			return $actions;
		}

		if ($user->ID === get_current_user_id()) {
			return $actions;
		}

		if ($this->isImpersonating()) {
			return $actions;
		}

		$switchUrl = wp_nonce_url(
			admin_url('admin.php?action=sctk_switch_user&user_id=' . $user->ID),
			'sctk_switch_user_' . $user->ID
		);

		$actions['sctk_switch'] = sprintf(
			'<a href="%s">%s</a>',
			esc_url($switchUrl),
			esc_html__('Switch to', Constants::TEXT_DOMAIN)
		);

		return $actions;
	}

	/**
	 * Render a prominent "Switch to" button on the user edit profile page.
	 *
	 * Displays a styled banner with a switch button when viewing
	 * another user's profile, guarded by screen, capability, and impersonation checks.
	 */
	public function renderProfileSwitchButton(): void
	{
		$screen = get_current_screen();
		if (null === $screen || 'user-edit' !== $screen->id) {
			return;
		}

		if (!current_user_can('manage_options') || $this->isImpersonating()) {
			return;
		}

		$userId = isset($_GET['user_id']) ? absint($_GET['user_id']) : 0;
		if (0 === $userId || $userId === get_current_user_id()) {
			return;
		}

		$user = get_userdata($userId);
		if (false === $user) {
			return;
		}

		$switchUrl = wp_nonce_url(
			admin_url('admin.php?action=sctk_switch_user&user_id=' . $user->ID),
			'sctk_switch_user_' . $user->ID
		);

		?>
		<div id="sctk-profile-switch">
			<style>
				#sctk-profile-switch {
					margin: 20px 0;
					padding: 14px 20px;
					background: #fff;
					border: 1px solid #c3c4c7;
					border-left: 4px solid #b8642a;
					border-radius: 2px;
					display: flex;
					align-items: center;
					justify-content: space-between;
				}
				#sctk-profile-switch span {
					font-size: 13px;
					color: #50575e;
				}
				#sctk-profile-switch strong {
					color: #1d2327;
				}
				#sctk-profile-switch a {
					display: inline-block;
					padding: 6px 16px;
					background: #b8642a;
					color: #fff;
					text-decoration: none;
					border-radius: 3px;
					font-weight: 600;
					font-size: 13px;
				}
				#sctk-profile-switch a:hover {
					background: #a05724;
					color: #fff;
				}
			</style>
			<span>
				<?php
				printf(
					/* translators: %s: user display name */
					esc_html__('Switch to %s and browse the site as this user.', Constants::TEXT_DOMAIN),
					'<strong>' . esc_html($user->display_name) . '</strong>'
				);
				?>
			</span>
			<a href="<?php echo esc_url($switchUrl); ?>">
				<?php
				printf(
					/* translators: %s: user display name */
					esc_html__('Switch to %s', Constants::TEXT_DOMAIN),
					esc_html($user->display_name)
				);
				?>
			</a>
		</div>
		<?php
	}

	/**
	 * Handle switching to another user.
	 *
	 * Validates permissions, nonce, and target user before
	 * performing the user switch.
	 */
	public function handleSwitch(): void
	{
		if (!isset($_GET['action'], $_GET['user_id']) || 'sctk_switch_user' !== $_GET['action']) {
			return;
		}

		if (!current_user_can('manage_options')) {
			wp_die(
				esc_html__('You do not have permission to switch users.', Constants::TEXT_DOMAIN),
				esc_html__('Unauthorized', Constants::TEXT_DOMAIN),
				['response' => 403]
			);
		}

		if ($this->isImpersonating()) {
			wp_die(
				esc_html__('You cannot switch users while already impersonating.', Constants::TEXT_DOMAIN),
				esc_html__('Switch Not Allowed', Constants::TEXT_DOMAIN),
				['response' => 403]
			);
		}

		$targetId = (int) $_GET['user_id'];

		check_admin_referer('sctk_switch_user_' . $targetId);

		$targetUser = get_userdata($targetId);
		if (false === $targetUser) {
			wp_die(
				esc_html__('Target user does not exist.', Constants::TEXT_DOMAIN),
				esc_html__('User Not Found', Constants::TEXT_DOMAIN),
				['response' => 404]
			);
		}

		$currentUserId = get_current_user_id();

		update_user_meta($targetId, self::ORIGINAL_USER_META, $currentUserId);

		$token = self::generateToken($currentUserId, $targetId);

		wp_clear_auth_cookie();
		wp_set_auth_cookie($targetId);
		wp_set_current_user($targetId);

		nocache_headers();
		wp_safe_redirect(add_query_arg([
			'sctk_switched' => time(),
			'sctk_token'    => $token,
		], admin_url('profile.php')));
		exit;
	}

	/**
	 * Handle switching back to the original user.
	 *
	 * Hooked to admin_post_sctk_switch_back, accessible by all
	 * logged-in users regardless of capability. Validates impersonation
	 * state, nonce, and original user before restoring the session.
	 */
	public function handleSwitchBack(): void
	{
		if (!$this->isImpersonating()) {
			wp_die(
				esc_html__('You are not currently impersonating a user.', Constants::TEXT_DOMAIN),
				esc_html__('Not Impersonating', Constants::TEXT_DOMAIN),
				['response' => 403]
			);
		}

		$originalId = $this->getOriginalUserId();
		if (null === $originalId) {
			wp_die(
				esc_html__('Could not determine original user.', Constants::TEXT_DOMAIN),
				esc_html__('Switch Back Error', Constants::TEXT_DOMAIN),
				['response' => 500]
			);
		}

		$providedToken = isset($_GET['_token']) ? sanitize_text_field(wp_unslash($_GET['_token'])) : '';
		$expectedToken = self::generateToken($originalId, get_current_user_id());

		if (!hash_equals($expectedToken, $providedToken)) {
			wp_die(
				esc_html__('Invalid switch-back token.', Constants::TEXT_DOMAIN),
				esc_html__('Switch Back Error', Constants::TEXT_DOMAIN),
				['response' => 403]
			);
		}

		$originalUser = get_userdata($originalId);
		if (false === $originalUser || !$originalUser->has_cap('manage_options')) {
			wp_die(
				esc_html__('Original user no longer has sufficient permissions.', Constants::TEXT_DOMAIN),
				esc_html__('Switch Back Error', Constants::TEXT_DOMAIN),
				['response' => 403]
			);
		}

		delete_user_meta(get_current_user_id(), self::ORIGINAL_USER_META);

		wp_clear_auth_cookie();
		wp_set_auth_cookie($originalId);
		wp_set_current_user($originalId);

		$redirect = wp_get_referer() ?: admin_url();
		$redirect = add_query_arg('sctk_switched', time(), $redirect);

		nocache_headers();
		wp_safe_redirect($redirect);
		exit;
	}

	/**
	 * Render floating switch-back bar when impersonating.
	 *
	 * Displays a slim fixed bar at the bottom of the viewport on all
	 * pages (admin and frontend), showing who is being impersonated
	 * and a link to switch back.
	 */
	public function renderSwitchBackBar(): void
	{
		if (!$this->isImpersonating()) {
			return;
		}

		$originalId = $this->getOriginalUserId();
		if (null === $originalId) {
			return;
		}

		$originalUser = get_userdata($originalId);
		if (false === $originalUser) {
			return;
		}

		$currentUser = wp_get_current_user();

		$token = isset($_GET['sctk_token']) ? sanitize_text_field(wp_unslash($_GET['sctk_token'])) : '';
		if ('' === $token) {
			$token = self::generateToken($originalId, get_current_user_id());
		}

		$switchBackUrl = add_query_arg(
			'_token',
			$token,
			admin_url('admin-post.php?action=sctk_switch_back')
		);

		?>
		<div id="sctk-switch-back-bar">
			<style>
				#sctk-switch-back-bar {
					position: fixed;
					bottom: 0;
					left: 0;
					right: 0;
					z-index: 99999;
					background: #20292e;
					border-top: 2px solid #b8642a;
					padding: 8px 20px;
					text-align: center;
					font-size: 13px;
					font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif;
					color: #c3c4c7;
					line-height: 1.4;
				}
				#sctk-switch-back-bar strong {
					color: #fff;
				}
				#sctk-switch-back-bar a {
					display: inline-block;
					margin-left: 10px;
					padding: 2px 12px;
					background: #b8642a;
					color: #fff;
					text-decoration: none;
					border-radius: 3px;
					font-weight: 600;
					font-size: 12px;
				}
				#sctk-switch-back-bar a:hover {
					background: #a05724;
				}
			</style>
			<?php
			printf(
				/* translators: 1: current user display name (wrapped in <strong>), 2: switch back link */
				__('Logged in as %1$s %2$s', Constants::TEXT_DOMAIN),
				'<strong>' . esc_html($currentUser->display_name) . '</strong>',
				sprintf(
					'<a href="%s">%s</a>',
					esc_url($switchBackUrl),
					esc_html(
						sprintf(
							/* translators: %s: original user display name */
							__('Switch back to %s', Constants::TEXT_DOMAIN),
							$originalUser->display_name
						)
					)
				)
			);
			?>
		</div>
		<?php
	}

	/**
	 * Generate a deterministic HMAC token for switch-back verification.
	 *
	 * Uses the original and target user IDs combined with the site's
	 * nonce salt to produce a consistent, unforgeable token.
	 *
	 * @param int $originalUserId The original admin user ID.
	 * @param int $targetUserId   The impersonated user ID.
	 * @return string HMAC token.
	 */
	private static function generateToken(int $originalUserId, int $targetUserId): string
	{
		return hash_hmac('sha256', $originalUserId . ':' . $targetUserId, wp_salt('nonce'));
	}

	/**
	 * Check if the current user is impersonating another user.
	 *
	 * @return bool True if currently impersonating.
	 */
	private function isImpersonating(): bool
	{
		$originalId = get_user_meta(get_current_user_id(), self::ORIGINAL_USER_META, true);

		return '' !== $originalId && false !== $originalId;
	}

	/**
	 * Get the original user ID from impersonation meta.
	 *
	 * @return int|null Original user ID, or null if not impersonating.
	 */
	private function getOriginalUserId(): ?int
	{
		$originalId = get_user_meta(get_current_user_id(), self::ORIGINAL_USER_META, true);

		if ('' === $originalId || false === $originalId) {
			return null;
		}

		return (int) $originalId;
	}
}
