<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features\DevTools;

use SignocoreToolkit\Application\Constants;
use SignocoreToolkit\Features\DevTools\Traits\HasDevToolsTabs;
use SignocoreToolkit\Infrastructure\Traits\Singleton;

/**
 * Transient management feature.
 *
 * @package SignocoreToolkit\Features\DevTools
 * @since 3.0.0
 */
final class TransientViewer
{
	use HasDevToolsTabs;
	use Singleton;

	/**
	 * Number of transients to display per page.
	 */
	private int $perPage = 30;

	/**
	 * Initialize transient manager features.
	 *
	 * Registers admin handlers for transient deletion operations.
	 */
	protected function init(): void
	{
		add_action('admin_init', [$this, 'handleDelete']);
		add_action('admin_init', [$this, 'handleBulkDelete']);
		add_action('admin_init', [$this, 'handleDeleteExpired']);
	}

	/**
	 * Render the transient management admin page.
	 *
	 * Displays a paginated, searchable table of all transients with
	 * options to delete individual, selected, or expired transients.
	 */
	public function renderPage(): void
	{
		$search = isset($_GET['s']) ? sanitize_text_field(wp_unslash($_GET['s'])) : '';
		$currentPage = isset($_GET['paged']) ? max(1, (int) $_GET['paged']) : 1;
		$totalCount = $this->getTotalCount($search);
		$transients = $this->getTransients($currentPage, $search);
		$totalPages = (int) ceil($totalCount / $this->perPage);

		?>
		<div class="wrap">
			<h1 class="wp-heading-inline">
				<?php echo esc_html__('Transients', Constants::TEXT_DOMAIN); ?>
				<span class="title-count theme-count"><?php echo esc_html((string) $totalCount); ?></span>
			</h1>
			<hr class="wp-header-end">

			<?php
			$this->renderMainTabs();
			$this->renderDevToolsTabs('signocore-toolkit-transients');
			?>

			<?php if (isset($_GET['deleted'])) : ?>
				<div class="notice notice-success is-dismissible">
					<p>
						<?php
						printf(
							/* translators: %d: number of deleted transients */
							esc_html__('%d transient(s) deleted.', Constants::TEXT_DOMAIN),
							(int) $_GET['deleted']
						);
						?>
					</p>
				</div>
			<?php endif; ?>

			<form method="post" style="float:right;margin:0 0 10px 6px;">
				<?php wp_nonce_field('sctk_delete_expired_transients', 'sctk_delete_expired_transients_nonce'); ?>
				<input type="hidden" name="action" value="sctk_delete_expired_transients" />
				<input
					type="submit"
					class="button button-link-delete"
					value="<?php echo esc_attr__('Delete All Expired', Constants::TEXT_DOMAIN); ?>"
				>
			</form>

			<form method="get" class="sctk-search-box">
				<input type="hidden" name="page" value="<?php echo esc_attr(sanitize_text_field(wp_unslash($_GET['page'] ?? ''))); ?>" />
				<label class="screen-reader-text" for="transient-search-input">
					<?php echo esc_html__('Search Transients', Constants::TEXT_DOMAIN); ?>
				</label>
				<input
					type="search"
					id="transient-search-input"
					name="s"
					value="<?php echo esc_attr($search); ?>"
					placeholder="<?php echo esc_attr__('Search by name...', Constants::TEXT_DOMAIN); ?>"
				/>
				<input
					type="submit"
					class="button"
					value="<?php echo esc_attr__('Search', Constants::TEXT_DOMAIN); ?>"
				/>
			</form>

			<form method="post">
				<?php wp_nonce_field('sctk_bulk_delete_transients', 'sctk_bulk_delete_transients_nonce'); ?>
				<input type="hidden" name="action" value="sctk_bulk_delete_transients" />

				<table class="widefat striped">
					<thead>
						<tr>
							<td class="manage-column column-cb check-column">
								<input type="checkbox" id="cb-select-all" />
							</td>
							<th scope="col"><?php echo esc_html__('Name', Constants::TEXT_DOMAIN); ?></th>
							<th scope="col"><?php echo esc_html__('Value', Constants::TEXT_DOMAIN); ?></th>
							<th scope="col"><?php echo esc_html__('Expiration', Constants::TEXT_DOMAIN); ?></th>
							<th scope="col"><?php echo esc_html__('Size', Constants::TEXT_DOMAIN); ?></th>
						</tr>
					</thead>
					<tbody>
						<?php if ([] === $transients) : ?>
							<tr>
								<td colspan="5"><?php echo esc_html__('No transients found.', Constants::TEXT_DOMAIN); ?></td>
							</tr>
						<?php else : ?>
							<?php foreach ($transients as $transient) : ?>
								<?php
								$deleteUrl = wp_nonce_url(
									add_query_arg([
										'action' => 'sctk_delete_transient',
										'transient' => $transient->name,
									]),
									'sctk_delete_transient_' . $transient->name
								);
								?>
								<tr>
									<th scope="row" class="check-column">
										<input
											type="checkbox"
											name="transient_names[]"
											value="<?php echo esc_attr($transient->name); ?>"
										/>
									</th>
									<td>
										<strong><?php echo esc_html($transient->name); ?></strong>
										<div class="row-actions">
											<span class="delete">
												<a href="<?php echo esc_url($deleteUrl); ?>" class="submitdelete" aria-label="<?php echo esc_attr(sprintf(
													/* translators: %s: transient name */
													__('Delete transient: %s', Constants::TEXT_DOMAIN),
													$transient->name
												)); ?>">
													<?php echo esc_html__('Delete', Constants::TEXT_DOMAIN); ?>
												</a>
											</span>
										</div>
									</td>
									<td>
										<code><?php echo esc_html($this->formatValue($transient->value)); ?></code>
									</td>
									<td>
										<?php if (null === $transient->expiration) : ?>
											<?php echo esc_html__('Never', Constants::TEXT_DOMAIN); ?>
										<?php elseif ($transient->expiration < time()) : ?>
											<span style="color: #dc3232;" title="<?php echo esc_attr((string) wp_date('Y-m-d H:i:s', $transient->expiration)); ?>">
												<?php echo esc_html__('Expired', Constants::TEXT_DOMAIN); ?>
												(<?php echo esc_html(human_time_diff($transient->expiration)); ?>)
											</span>
										<?php else : ?>
											<span title="<?php echo esc_attr((string) wp_date('Y-m-d H:i:s', $transient->expiration)); ?>">
												<?php echo esc_html(human_time_diff(time(), $transient->expiration)); ?>
											</span>
										<?php endif; ?>
									</td>
									<td><?php echo esc_html($this->getSize($transient->value)); ?></td>
								</tr>
							<?php endforeach; ?>
						<?php endif; ?>
					</tbody>
					<tfoot>
						<tr>
							<td class="manage-column column-cb check-column">
								<input type="checkbox" id="cb-select-all-footer" />
							</td>
							<th scope="col"><?php echo esc_html__('Name', Constants::TEXT_DOMAIN); ?></th>
							<th scope="col"><?php echo esc_html__('Value', Constants::TEXT_DOMAIN); ?></th>
							<th scope="col"><?php echo esc_html__('Expiration', Constants::TEXT_DOMAIN); ?></th>
							<th scope="col"><?php echo esc_html__('Size', Constants::TEXT_DOMAIN); ?></th>
						</tr>
					</tfoot>
				</table>

				<div class="tablenav bottom">
					<div class="alignleft actions bulkactions">
						<?php
						submit_button(
							esc_html__('Delete Selected', Constants::TEXT_DOMAIN),
							'secondary',
							'submit',
							false
						);
						?>
					</div>
					<div class="tablenav-pages">
						<?php
						$paginationLinks = paginate_links([
							'base' => add_query_arg('paged', '%#%'),
							'format' => '',
							'prev_text' => '&laquo;',
							'next_text' => '&raquo;',
							'total' => $totalPages,
							'current' => $currentPage,
						]);

						if (null !== $paginationLinks) {
							echo wp_kses_post($paginationLinks);
						}
						?>
					</div>
				</div>
			</form>
		</div>
		<?php
	}

	/**
	 * Handle single transient deletion.
	 *
	 * Processes GET requests to delete an individual transient
	 * after verifying nonce and capabilities.
	 */
	public function handleDelete(): void
	{
		if (!isset($_GET['action'], $_GET['transient'])) {
			return;
		}

		if ('sctk_delete_transient' !== $_GET['action']) {
			return;
		}

		$transientName = sanitize_text_field(wp_unslash($_GET['transient']));

		check_admin_referer('sctk_delete_transient_' . $transientName);

		if (!current_user_can('manage_options')) {
			wp_die(
				esc_html__('You do not have permission to perform this action.', Constants::TEXT_DOMAIN),
				'',
				['response' => 403]
			);
		}

		delete_transient($transientName);

		wp_safe_redirect(add_query_arg('deleted', 1, remove_query_arg(['action', 'transient', '_wpnonce'])));
		exit;
	}

	/**
	 * Handle bulk transient deletion.
	 *
	 * Processes POST requests to delete multiple selected transients
	 * after verifying nonce and capabilities.
	 */
	public function handleBulkDelete(): void
	{
		if (!isset($_POST['action'])) {
			return;
		}

		if ('sctk_bulk_delete_transients' !== $_POST['action']) {
			return;
		}

		check_admin_referer('sctk_bulk_delete_transients', 'sctk_bulk_delete_transients_nonce');

		if (!current_user_can('manage_options')) {
			wp_die(
				esc_html__('You do not have permission to perform this action.', Constants::TEXT_DOMAIN),
				'',
				['response' => 403]
			);
		}

		$transientNames = isset($_POST['transient_names']) && is_array($_POST['transient_names'])
			? array_map('sanitize_text_field', wp_unslash($_POST['transient_names']))
			: [];

		$deleted = 0;
		foreach ($transientNames as $name) {
			if (delete_transient($name)) {
				$deleted++;
			}
		}

		wp_safe_redirect(add_query_arg('deleted', $deleted, remove_query_arg(['action'])));
		exit;
	}

	/**
	 * Handle expired transient deletion.
	 *
	 * Queries the database for expired transient timeouts and deletes
	 * them after verifying nonce and capabilities.
	 */
	public function handleDeleteExpired(): void
	{
		if (!isset($_POST['action'])) {
			return;
		}

		if ('sctk_delete_expired_transients' !== $_POST['action']) {
			return;
		}

		check_admin_referer('sctk_delete_expired_transients', 'sctk_delete_expired_transients_nonce');

		if (!current_user_can('manage_options')) {
			wp_die(
				esc_html__('You do not have permission to perform this action.', Constants::TEXT_DOMAIN),
				'',
				['response' => 403]
			);
		}

		global $wpdb;

		$expired = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE %s AND option_value < %d",
				'_transient_timeout_%',
				time()
			)
		);

		$deleted = 0;
		foreach ($expired as $optionName) {
			$transientName = str_replace('_transient_timeout_', '', $optionName);
			if (delete_transient($transientName)) {
				$deleted++;
			}
		}

		wp_safe_redirect(add_query_arg('deleted', $deleted, remove_query_arg(['action'])));
		exit;
	}

	/**
	 * Get paginated list of transients.
	 *
	 * Queries the options table for transient entries, optionally
	 * filtered by search term, with pagination support.
	 *
	 * @param int    $page   Current page number.
	 * @param string $search Search term to filter transient names.
	 * @return array<int, object{name: string, value: string, expiration: ?int, size: string}> Transient data objects.
	 */
	private function getTransients(int $page, string $search): array
	{
		global $wpdb;

		$offset = ($page - 1) * $this->perPage;

		if ('' !== $search) {
			$results = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT option_name, option_value FROM {$wpdb->options}
					WHERE option_name LIKE %s
					AND option_name NOT LIKE %s
					AND option_name LIKE %s
					ORDER BY option_name ASC
					LIMIT %d OFFSET %d",
					'_transient_%',
					'_transient_timeout_%',
					'%' . $wpdb->esc_like($search) . '%',
					$this->perPage,
					$offset
				)
			);
		} else {
			$results = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT option_name, option_value FROM {$wpdb->options}
					WHERE option_name LIKE %s
					AND option_name NOT LIKE %s
					ORDER BY option_name ASC
					LIMIT %d OFFSET %d",
					'_transient_%',
					'_transient_timeout_%',
					$this->perPage,
					$offset
				)
			);
		}

		$transients = [];
		foreach ($results as $row) {
			$name = str_replace('_transient_', '', $row->option_name);
			$transients[] = (object) [
				'name' => $name,
				'value' => $row->option_value,
				'expiration' => $this->getTransientExpiration($name),
				'size' => $this->getSize($row->option_value),
			];
		}

		return $transients;
	}

	/**
	 * Get the expiration timestamp for a transient.
	 *
	 * Looks up the timeout option for the given transient name.
	 *
	 * @param string $name Transient name (without prefix).
	 * @return int|null Expiration timestamp, or null if no expiration.
	 */
	private function getTransientExpiration(string $name): ?int
	{
		global $wpdb;

		$timeout = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT option_value FROM {$wpdb->options} WHERE option_name = %s",
				'_transient_timeout_' . $name
			)
		);

		if (null === $timeout) {
			return null;
		}

		return (int) $timeout;
	}

	/**
	 * Get total count of transients matching the search criteria.
	 *
	 * @param string $search Search term to filter transient names.
	 * @return int Total number of matching transients.
	 */
	private function getTotalCount(string $search): int
	{
		global $wpdb;

		if ('' !== $search) {
			$count = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM {$wpdb->options}
					WHERE option_name LIKE %s
					AND option_name NOT LIKE %s
					AND option_name LIKE %s",
					'_transient_%',
					'_transient_timeout_%',
					'%' . $wpdb->esc_like($search) . '%'
				)
			);
		} else {
			$count = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM {$wpdb->options}
					WHERE option_name LIKE %s
					AND option_name NOT LIKE %s",
					'_transient_%',
					'_transient_timeout_%'
				)
			);
		}

		return (int) $count;
	}

	/**
	 * Format a transient value for display.
	 *
	 * Truncates long values to 100 characters and escapes for safe output.
	 *
	 * @param string $value Raw transient value.
	 * @return string Formatted and escaped value.
	 */
	private function formatValue(string $value): string
	{
		if (100 < strlen($value)) {
			$value = substr($value, 0, 100) . '...';
		}

		return esc_html($value);
	}

	/**
	 * Get human-readable size of a value.
	 *
	 * @param string $value The value to measure.
	 * @return string Human-readable file size.
	 */
	private function getSize(string $value): string
	{
		return size_format(strlen($value));
	}
}
