<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features\DevTools;

use SignocoreToolkit\Application\Constants;
use SignocoreToolkit\Infrastructure\Traits\Singleton;
use SignocoreToolkit\Infrastructure\Traits\ZipDownload;

/**
 * Theme download functionality.
 *
 * @package SignocoreToolkit\Features\DevTools
 * @since 2.1.0
 */
final class ThemeDownload
{
	use Singleton;
	use ZipDownload;

	/**
	 * Link color for download actions.
	 */
	private string $color = '#546e7a';

	/**
	 * Initialize theme download features.
	 *
	 * Sets up theme data injection, JS button rendering,
	 * and download handler.
	 */
	protected function init(): void
	{
		add_filter('wp_prepare_themes_for_js', [$this, 'addDownloadData']);
		add_action('admin_footer-themes.php', [$this, 'renderDownloadButton']);
		add_action('admin_init', [$this, 'handleDownloadRequest']);
	}

	/**
	 * Add download URL to each theme's JavaScript data.
	 *
	 * @param array<string, array<string, mixed>> $themes Prepared theme data.
	 * @return array<string, array<string, mixed>> Modified theme data.
	 */
	public function addDownloadData(array $themes): array
	{
		if (!current_user_can('manage_options')) {
			return $themes;
		}

		foreach ($themes as $stylesheet => &$data) {
			$data['downloadUrl'] = add_query_arg([
				'action' => 'signocore_download_theme',
				'theme' => $stylesheet,
				'_wpnonce' => wp_create_nonce('signocore_download_theme_' . $stylesheet),
			], admin_url('themes.php'));
		}

		return $themes;
	}

	/**
	 * Render inline JavaScript for the download button in theme overlay.
	 *
	 * Observes the theme detail overlay and injects a download button
	 * into the theme actions area.
	 */
	public function renderDownloadButton(): void
	{
		if (!current_user_can('manage_options')) {
			return;
		}

		$label = esc_js(__('Download', Constants::TEXT_DOMAIN));
		$ariaLabel = esc_js(__('Download Theme', Constants::TEXT_DOMAIN));
		$color = esc_js($this->color);

		?>
		<script>
		(function() {
			var icon = '<svg xmlns="http://www.w3.org/2000/svg" width="14" height="13" viewBox="0 0 256 240" fill="currentColor" style="vertical-align: -2px;"><g transform="translate(0,240) scale(0.1,-0.1)" stroke="none"><path d="M1210 2378c-83-43-80-16-80-655 0-342-4-563-9-563-5 0-94 90-196 199-202 216-232 238-307 226-70-12-128-82-128-153 0-16 10-47 22-70 29-51 660-728 699-747 39-21 108-19 146 4 39 25 672 700 696 744 48 89-12 207-115 223-67 11-108-20-308-233l-185-197-5 564c-5 544-6 566-25 600-22 38-93 80-135 80-14 0-46-10-70-22z"/><path d="M91 882c-43-21-70-57-81-108-13-54-12-344 1-423 29-179 161-311 340-340 88-15 1770-15 1859 0 176 28 308 155 340 328 5 29 10 138 10 240 0 174-2 190-22 231-42 83-130 113-208 71-68-36-74-60-80-298l-5-213-28-27-27-28-881-3c-600-2-893 1-919 8-70 19-74 35-80 276-5 195-7 215-26 241-26 36-79 63-124 63-19 0-50-8-69-18z"/></g></svg>';
			var themes = window._wpThemeSettings && window._wpThemeSettings.themes ? window._wpThemeSettings.themes : [];

			function getSlugFromUrl() {
				return new URLSearchParams(window.location.search).get('theme');
			}

			function getTheme(slug) {
				for (var i = 0; i < themes.length; i++) {
					if (themes[i].id === slug) {
						return themes[i];
					}
				}
				return null;
			}

			function addButton() {
				var overlay = document.querySelector('.theme-overlay .theme-actions');
				if (!overlay) {
					return;
				}

				// Remove existing button (overlay is reused for different themes)
				var existing = overlay.querySelector('.signocore-download-theme');
				if (existing) {
					existing.remove();
				}

				var slug = getSlugFromUrl();
				if (!slug) {
					return;
				}

				var theme = getTheme(slug);
				if (!theme || !theme.downloadUrl) {
					return;
				}

				var link = document.createElement('a');
				link.href = theme.downloadUrl;
				link.className = 'button signocore-download-theme';
				link.setAttribute('aria-label', '<?php echo $ariaLabel; ?>');
				link.style.cssText = 'color: <?php echo $color; ?>; border-color: <?php echo $color; ?>; margin-left: 8px;';
				link.innerHTML = icon + ' <?php echo $label; ?>';

				overlay.appendChild(link);
			}

			// Observe URL changes when overlay opens or navigates between themes
			var lastTheme = getSlugFromUrl();
			var observer = new MutationObserver(function() {
				var currentTheme = getSlugFromUrl();
				if (currentTheme && currentTheme !== lastTheme) {
					lastTheme = currentTheme;
					addButton();
				} else if (currentTheme && !document.querySelector('.signocore-download-theme')) {
					addButton();
				}
			});

			observer.observe(document.body, { childList: true, subtree: true });

			// Handle overlay already visible on page load
			addButton();
		})();
		</script>
		<?php
	}

	/**
	 * Handle theme download request.
	 *
	 * Processes download requests, validates permissions and nonces,
	 * then generates and streams a zip file of the requested theme.
	 */
	public function handleDownloadRequest(): void
	{
		if (!$this->isValidDownloadRequest()) {
			return;
		}

		$stylesheet = sanitize_text_field(wp_unslash($_GET['theme']));

		check_admin_referer('signocore_download_theme_' . $stylesheet);

		$theme = wp_get_theme($stylesheet);
		if (!$theme->exists()) {
			wp_die(
				esc_html__('Invalid theme.', Constants::TEXT_DOMAIN),
				esc_html__('Theme Download Error', Constants::TEXT_DOMAIN),
				['response' => 404]
			);
		}

		$themeVersion = $theme->get('Version');
		$zipName = sanitize_file_name($stylesheet) . '-' . $themeVersion . '.zip';

		$this->createAndStreamZip($theme->get_stylesheet_directory(), $zipName);
	}

	/**
	 * Validate download request parameters.
	 *
	 * @return bool True if request is valid.
	 */
	private function isValidDownloadRequest(): bool
	{
		return isset($_GET['action'], $_GET['theme'])
			&& 'signocore_download_theme' === $_GET['action']
			&& current_user_can('manage_options');
	}
}
