<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features\DevTools;

use SignocoreToolkit\Application\Constants;
use SignocoreToolkit\Features\DevTools\Traits\HasDevToolsTabs;
use SignocoreToolkit\Infrastructure\Traits\Singleton;

/**
 * System status page.
 *
 * @package SignocoreToolkit\Features\DevTools
 * @since 3.0.0
 */
final class SystemStatus
{
	use HasDevToolsTabs;
	use Singleton;

	/**
	 * Initialize system status features.
	 *
	 * Registers the AJAX handler for deleting inactive plugins.
	 */
	protected function init(): void
	{
		add_action('wp_ajax_sctk_delete_inactive_plugins', [$this, 'handleDeleteInactivePlugins']);
		add_action('wp_ajax_sctk_delete_inactive_themes', [$this, 'handleDeleteInactiveThemes']);
	}

	/**
	 * Render the system status admin page.
	 *
	 * Outputs all status sections as styled cards with status icons
	 * and interactive features (copy, delete inactive plugins).
	 */
	public function renderPage(): void
	{
		$deleteNonce = wp_create_nonce('sctk_delete_inactive_plugins');
		$deleteThemesNonce = wp_create_nonce('sctk_delete_inactive_themes');
		?>
		<div class="wrap">
			<h1><?php echo esc_html__('System Status', Constants::TEXT_DOMAIN); ?></h1>
			<hr class="wp-header-end">

			<?php
			$this->renderMainTabs();
			$this->renderDevToolsTabs('signocore-toolkit-status');
			?>

			<p style="margin:10px 0 15px;">
				<button type="button" class="button sctk-copy-status">
					<?php echo esc_html__('Copy to Clipboard', Constants::TEXT_DOMAIN); ?>
				</button>
			</p>

			<?php $this->renderStyles(); ?>

			<?php $this->renderWordPressEnvironment(); ?>
			<?php $this->renderServerEnvironment(); ?>
			<?php $this->renderDatabase(); ?>
			<?php $this->renderActivePlugins(); ?>
			<?php $this->renderInactivePlugins($deleteNonce); ?>
			<?php $this->renderMustUsePlugins(); ?>
			<?php $this->renderTheme(); ?>
			<?php $this->renderInactiveThemes($deleteThemesNonce); ?>
		</div>

		<?php $this->renderScripts($deleteNonce); ?>
		<?php
	}

	/**
	 * Render the WordPress environment section.
	 */
	private function renderWordPressEnvironment(): void
	{
		$isProduction = 'production' === $this->getEnvironmentType();
		$wpDebug = defined('WP_DEBUG') && WP_DEBUG;
		$wpDebugLog = defined('WP_DEBUG_LOG') && WP_DEBUG_LOG;
		$scriptDebug = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG;
		$cronDisabled = defined('DISABLE_WP_CRON') && DISABLE_WP_CRON;
		$isSsl = is_ssl();
		$objectCache = wp_using_ext_object_cache();
		$memoryLimit = defined('WP_MEMORY_LIMIT') ? WP_MEMORY_LIMIT : '40M';
		$memoryLimitBytes = wp_convert_hr_to_bytes($memoryLimit);

		$rows = [
			[
				'label' => __('WordPress Version', Constants::TEXT_DOMAIN),
				'value' => get_bloginfo('version'),
			],
			[
				'label' => __('Site URL', Constants::TEXT_DOMAIN),
				'value' => site_url(),
			],
			[
				'label' => __('Home URL', Constants::TEXT_DOMAIN),
				'value' => home_url(),
			],
			[
				'label' => __('HTTPS', Constants::TEXT_DOMAIN),
				'value' => $isSsl
					? $this->statusGood(__('Yes', Constants::TEXT_DOMAIN))
					: $this->statusBad(__('No', Constants::TEXT_DOMAIN)),
			],
			[
				'label' => __('Environment Type', Constants::TEXT_DOMAIN),
				'value' => $this->getEnvironmentTypeDisplay(),
			],
			[
				'label' => __('Multisite', Constants::TEXT_DOMAIN),
				'value' => is_multisite()
					? __('Yes', Constants::TEXT_DOMAIN)
					: __('No', Constants::TEXT_DOMAIN),
			],
			[
				'label' => __('Permalink Structure', Constants::TEXT_DOMAIN),
				'value' => get_option('permalink_structure') ?: __('Default', Constants::TEXT_DOMAIN),
			],
			[
				'label' => __('WP Debug', Constants::TEXT_DOMAIN),
				'value' => $wpDebug
					? ($isProduction ? $this->statusWarning(__('Enabled', Constants::TEXT_DOMAIN)) : $this->statusGood(__('Enabled', Constants::TEXT_DOMAIN)))
					: __('Disabled', Constants::TEXT_DOMAIN),
			],
			[
				'label' => __('WP Debug Log', Constants::TEXT_DOMAIN),
				'value' => $wpDebugLog
					? ($isProduction ? $this->statusWarning(__('Enabled', Constants::TEXT_DOMAIN)) : $this->statusGood(__('Enabled', Constants::TEXT_DOMAIN)))
					: __('Disabled', Constants::TEXT_DOMAIN),
			],
			[
				'label' => __('Script Debug', Constants::TEXT_DOMAIN),
				'value' => $scriptDebug
					? ($isProduction ? $this->statusWarning(__('Enabled', Constants::TEXT_DOMAIN)) : __('Enabled', Constants::TEXT_DOMAIN))
					: __('Disabled', Constants::TEXT_DOMAIN),
			],
			[
				'label' => __('WP Memory Limit', Constants::TEXT_DOMAIN),
				'value' => $memoryLimitBytes < 67108864
					? $this->statusWarning($memoryLimit)
					: $memoryLimit,
			],
			[
				'label' => __('Max Upload Size', Constants::TEXT_DOMAIN),
				'value' => size_format(wp_max_upload_size()) ?: (string) __('Unknown', Constants::TEXT_DOMAIN),
			],
			[
				'label' => __('WP Cron', Constants::TEXT_DOMAIN),
				'value' => $cronDisabled
					? $this->statusBad(__('Disabled', Constants::TEXT_DOMAIN))
					: $this->statusGood(__('Enabled', Constants::TEXT_DOMAIN)),
			],
			[
				'label' => __('Object Cache', Constants::TEXT_DOMAIN),
				'value' => $objectCache
					? $this->statusGood(__('Active', Constants::TEXT_DOMAIN))
					: __('Inactive', Constants::TEXT_DOMAIN),
			],
		];

		$this->renderCard(__('WordPress Environment', Constants::TEXT_DOMAIN), $rows);
	}

	/**
	 * Render the server environment section.
	 */
	private function renderServerEnvironment(): void
	{
		global $wpdb;

		$phpVersion = PHP_VERSION;
		$phpVersionLow = version_compare($phpVersion, '8.1', '<');
		$curlVersionInfo = function_exists('curl_version') ? curl_version() : false;
		$curlVersion = is_array($curlVersionInfo) && isset($curlVersionInfo['version']) ? (string) $curlVersionInfo['version'] : null;

		$rows = [
			[
				'label' => __('PHP Version', Constants::TEXT_DOMAIN),
				'value' => $phpVersionLow
					? $this->statusWarning($phpVersion)
					: $phpVersion,
			],
			[
				'label' => __('Server Software', Constants::TEXT_DOMAIN),
				'value' => $_SERVER['SERVER_SOFTWARE'] ?? __('Unknown', Constants::TEXT_DOMAIN),
			],
			[
				'label' => __('MySQL Version', Constants::TEXT_DOMAIN),
				'value' => $wpdb->db_version(),
			],
			[
				'label' => __('Max Execution Time', Constants::TEXT_DOMAIN),
				'value' => (ini_get('max_execution_time') ?: '0') . 's',
			],
			[
				'label' => __('PHP Memory Limit', Constants::TEXT_DOMAIN),
				'value' => ini_get('memory_limit') ?: (string) __('Unknown', Constants::TEXT_DOMAIN),
			],
			[
				'label' => __('Upload Max Filesize', Constants::TEXT_DOMAIN),
				'value' => ini_get('upload_max_filesize') ?: (string) __('Unknown', Constants::TEXT_DOMAIN),
			],
			[
				'label' => __('Post Max Size', Constants::TEXT_DOMAIN),
				'value' => ini_get('post_max_size') ?: (string) __('Unknown', Constants::TEXT_DOMAIN),
			],
			[
				'label' => __('Max Input Vars', Constants::TEXT_DOMAIN),
				'value' => ini_get('max_input_vars') ?: (string) __('Unknown', Constants::TEXT_DOMAIN),
			],
			[
				'label' => __('cURL Version', Constants::TEXT_DOMAIN),
				'value' => null !== $curlVersion ? $curlVersion : (string) __('Not available', Constants::TEXT_DOMAIN),
			],
			[
				'label' => __('OpenSSL Version', Constants::TEXT_DOMAIN),
				'value' => defined('OPENSSL_VERSION_TEXT') ? OPENSSL_VERSION_TEXT : (string) __('Not available', Constants::TEXT_DOMAIN),
			],
		];

		$this->renderCard(__('Server Environment', Constants::TEXT_DOMAIN), $rows);
	}

	/**
	 * Render the database section.
	 */
	private function renderDatabase(): void
	{
		global $wpdb;

		$rows = [
			[
				'label' => __('Database Prefix', Constants::TEXT_DOMAIN),
				'value' => $wpdb->prefix,
			],
			[
				'label' => __('Database Size', Constants::TEXT_DOMAIN),
				'value' => $this->getDatabaseSize(),
			],
			[
				'label' => __('Autoloaded Options Size', Constants::TEXT_DOMAIN),
				'value' => $this->getAutoloadedOptionsSize(),
			],
			[
				'label' => __('Total Transients', Constants::TEXT_DOMAIN),
				'value' => (string) $this->getTransientCount(),
			],
		];

		$this->renderCard(__('Database', Constants::TEXT_DOMAIN), $rows);
	}

	/**
	 * Render the active plugins section.
	 */
	private function renderActivePlugins(): void
	{
		if (!function_exists('get_plugins')) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$allPlugins = get_plugins();
		$updates = get_site_transient('update_plugins');
		$activePlugins = [];

		foreach ($allPlugins as $file => $data) {
			if (is_plugin_active($file)) {
				$version = $data['Version'];

				if (is_object($updates) && isset($updates->response[$file])) {
					$version .= ' ' . $this->statusWarning(
						sprintf('→ %s', $updates->response[$file]->new_version)
					);
				}

				$activePlugins[] = [
					'label' => $data['Name'],
					'value' => $version,
				];
			}
		}

		$this->renderCard(
			sprintf(
				/* translators: %d: number of active plugins */
				__('Active Plugins (%d)', Constants::TEXT_DOMAIN),
				count($activePlugins)
			),
			$activePlugins
		);
	}

	/**
	 * Render the inactive plugins section with delete button.
	 *
	 * @param string $deleteNonce Nonce for the delete action.
	 */
	private function renderInactivePlugins(string $deleteNonce): void
	{
		if (!function_exists('get_plugins')) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$allPlugins = get_plugins();
		$updates = get_site_transient('update_plugins');
		$inactivePlugins = [];

		foreach ($allPlugins as $file => $data) {
			if (!is_plugin_active($file)) {
				$version = $data['Version'];

				if (is_object($updates) && isset($updates->response[$file])) {
					$version .= ' ' . $this->statusWarning(
						sprintf('→ %s', $updates->response[$file]->new_version)
					);
				}

				$inactivePlugins[] = [
					'label' => $data['Name'],
					'value' => $version,
				];
			}
		}

		$headerExtra = '';
		if ([] !== $inactivePlugins) {
			$headerExtra = sprintf(
				'<button type="button" class="button sctk-delete-inactive" data-nonce="%s">%s</button>',
				esc_attr($deleteNonce),
				esc_html__('Delete All Inactive', Constants::TEXT_DOMAIN)
			);
		}

		$this->renderCard(
			sprintf(
				/* translators: %d: number of inactive plugins */
				__('Inactive Plugins (%d)', Constants::TEXT_DOMAIN),
				count($inactivePlugins)
			),
			$inactivePlugins,
			$headerExtra
		);
	}

	/**
	 * Render the must-use plugins section.
	 */
	private function renderMustUsePlugins(): void
	{
		$muPlugins = get_mu_plugins();

		if ([] === $muPlugins) {
			return;
		}

		$rows = [];
		foreach ($muPlugins as $data) {
			$rows[] = [
				'label' => $data['Name'],
				'value' => $data['Version'] ?: '—',
			];
		}

		$this->renderCard(
			sprintf(
				/* translators: %d: number of must-use plugins */
				__('Must-Use Plugins (%d)', Constants::TEXT_DOMAIN),
				count($rows)
			),
			$rows
		);
	}

	/**
	 * Render the theme section.
	 */
	private function renderTheme(): void
	{
		$theme = wp_get_theme();
		$rows = [
			[
				'label' => __('Active Theme', Constants::TEXT_DOMAIN),
				'value' => $theme->get('Name') . ' ' . $theme->get('Version'),
			],
		];

		$parent = $theme->parent();
		if (false !== $parent) {
			$rows[] = [
				'label' => __('Parent Theme', Constants::TEXT_DOMAIN),
				'value' => $parent->get('Name') . ' ' . $parent->get('Version'),
			];
		}

		$rows[] = [
			'label' => __('Theme Path', Constants::TEXT_DOMAIN),
			'value' => get_stylesheet_directory(),
		];

		$this->renderCard(__('Theme', Constants::TEXT_DOMAIN), $rows);
	}

	/**
	 * Render the inactive themes section with delete button.
	 *
	 * Excludes the active theme and its parent theme from the list.
	 *
	 * @param string $deleteNonce Nonce for the delete action.
	 */
	private function renderInactiveThemes(string $deleteNonce): void
	{
		$activeTheme = wp_get_theme();
		$activeStylesheet = $activeTheme->get_stylesheet();
		$parentStylesheet = '';

		$parent = $activeTheme->parent();
		if (false !== $parent) {
			$parentStylesheet = $parent->get_stylesheet();
		}

		$allThemes = wp_get_themes();
		$inactiveThemes = [];

		foreach ($allThemes as $stylesheet => $theme) {
			if ($stylesheet === $activeStylesheet || $stylesheet === $parentStylesheet) {
				continue;
			}

			$inactiveThemes[] = [
				'label' => $theme->get('Name'),
				'value' => $theme->get('Version') ?: '—',
			];
		}

		$headerExtra = '';
		if ([] !== $inactiveThemes) {
			$headerExtra = sprintf(
				'<button type="button" class="button sctk-delete-inactive-themes" data-nonce="%s">%s</button>',
				esc_attr($deleteNonce),
				esc_html__('Delete All Inactive', Constants::TEXT_DOMAIN)
			);
		}

		$this->renderCard(
			sprintf(
				/* translators: %d: number of inactive themes */
				__('Inactive Themes (%d)', Constants::TEXT_DOMAIN),
				count($inactiveThemes)
			),
			$inactiveThemes,
			$headerExtra
		);
	}

	/**
	 * Render a status card section.
	 *
	 * @param string                              $title      Section title.
	 * @param array<int, array{label: string, value: string}> $rows       Label-value rows.
	 * @param string                              $headerExtra Optional HTML appended to the header.
	 */
	private function renderCard(string $title, array $rows, string $headerExtra = ''): void
	{
		?>
		<div class="sctk-status-card">
			<h2>
				<span><?php echo esc_html($title); ?></span>
				<?php
				if ('' !== $headerExtra) {
					// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- escaped at call site
					echo $headerExtra;
				}
				?>
			</h2>
			<table class="widefat">
				<tbody>
					<?php if ([] === $rows) : ?>
						<tr>
							<td colspan="2"><?php echo esc_html__('None', Constants::TEXT_DOMAIN); ?></td>
						</tr>
					<?php else : ?>
						<?php foreach ($rows as $row) : ?>
							<tr>
								<td><?php echo esc_html($row['label']); ?></td>
								<td><?php echo wp_kses($row['value'], $this->getAllowedStatusHtml()); ?></td>
							</tr>
						<?php endforeach; ?>
					<?php endif; ?>
				</tbody>
			</table>
		</div>
		<?php
	}

	/**
	 * Handle AJAX request to delete all inactive plugins.
	 */
	public function handleDeleteInactivePlugins(): void
	{
		if (!current_user_can('manage_options')) {
			wp_send_json_error(['message' => __('Unauthorized.', Constants::TEXT_DOMAIN)], 403);
		}

		check_ajax_referer('sctk_delete_inactive_plugins');

		if (!function_exists('get_plugins') || !function_exists('delete_plugins')) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}

		$allPlugins = get_plugins();
		$inactiveFiles = [];

		foreach ($allPlugins as $file => $data) {
			if (!is_plugin_active($file)) {
				$inactiveFiles[] = $file;
			}
		}

		if ([] === $inactiveFiles) {
			wp_send_json_error(['message' => __('No inactive plugins found.', Constants::TEXT_DOMAIN)], 404);
		}

		$result = delete_plugins($inactiveFiles);

		if (is_wp_error($result)) {
			wp_send_json_error(['message' => $result->get_error_message()], 500);
		}

		wp_send_json_success(['deleted' => count($inactiveFiles)]);
	}

	/**
	 * Handle AJAX request to delete all inactive themes.
	 *
	 * Preserves the active theme and its parent theme.
	 */
	public function handleDeleteInactiveThemes(): void
	{
		if (!current_user_can('manage_options')) {
			wp_send_json_error(['message' => __('Unauthorized.', Constants::TEXT_DOMAIN)], 403);
		}

		check_ajax_referer('sctk_delete_inactive_themes');

		if (!function_exists('delete_theme')) {
			require_once ABSPATH . 'wp-admin/includes/theme.php';
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}

		$activeTheme = wp_get_theme();
		$activeStylesheet = $activeTheme->get_stylesheet();
		$parentStylesheet = '';

		$parent = $activeTheme->parent();
		if (false !== $parent) {
			$parentStylesheet = $parent->get_stylesheet();
		}

		$allThemes = wp_get_themes();
		$deleted = 0;

		foreach ($allThemes as $stylesheet => $theme) {
			if ($stylesheet === $activeStylesheet || $stylesheet === $parentStylesheet) {
				continue;
			}

			delete_theme($stylesheet);
			$deleted++;
		}

		if (0 === $deleted) {
			wp_send_json_error(['message' => __('No inactive themes found.', Constants::TEXT_DOMAIN)], 404);
		}

		wp_send_json_success(['deleted' => $deleted]);
	}

	/**
	 * Format a value with a "good" status icon.
	 *
	 * @param string $text The value text.
	 * @return string HTML with status icon.
	 */
	private function statusGood(string $text): string
	{
		return '<span class="sctk-status-icon-good">&#10003;</span> ' . esc_html($text);
	}

	/**
	 * Format a value with a "warning" status icon.
	 *
	 * @param string $text The value text.
	 * @return string HTML with status icon.
	 */
	private function statusWarning(string $text): string
	{
		return '<span class="sctk-status-icon-warning">&#9888;</span> ' . esc_html($text);
	}

	/**
	 * Format a value with a "bad" status icon.
	 *
	 * @param string $text The value text.
	 * @return string HTML with status icon.
	 */
	private function statusBad(string $text): string
	{
		return '<span class="sctk-status-icon-bad">&#10007;</span> ' . esc_html($text);
	}

	/**
	 * Get allowed HTML tags for status icon output.
	 *
	 * @return array<string, array<string, bool>> Allowed HTML tags and attributes.
	 */
	private function getAllowedStatusHtml(): array
	{
		return [
			'span' => ['class' => true],
		];
	}

	/**
	 * Get the current environment type.
	 *
	 * @return string Environment type string.
	 */
	private function getEnvironmentType(): string
	{
		if (function_exists('wp_get_environment_type')) {
			return wp_get_environment_type();
		}

		return 'production';
	}

	/**
	 * Get environment type display value with override indicator.
	 *
	 * Shows the WordPress environment type and, if a toolkit override
	 * is active, appends a note showing the effective value.
	 *
	 * @return string HTML-safe display value.
	 */
	private function getEnvironmentTypeDisplay(): string
	{
		$wpType = $this->getEnvironmentType();
		$override = get_option('sctk_environment_override', 'auto');

		if ('auto' === $override || '' === $override) {
			return esc_html($wpType);
		}

		if ($override === $wpType) {
			return esc_html($wpType);
		}

		return esc_html($wpType)
			. ' <span class="sctk-status-override">'
			. sprintf(
				/* translators: %s: overridden environment name */
				esc_html__('(override: %s)', Constants::TEXT_DOMAIN),
				esc_html($override)
			)
			. '</span>';
	}

	/**
	 * Get total database size as a formatted string.
	 *
	 * @return string Formatted database size.
	 */
	private function getDatabaseSize(): string
	{
		global $wpdb;

		$result = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT SUM(data_length + index_length) FROM information_schema.TABLES WHERE table_schema = %s",
				DB_NAME
			)
		);

		if (null === $result) {
			return (string) __('Unknown', Constants::TEXT_DOMAIN);
		}

		$formatted = size_format((int) $result);

		return $formatted !== false ? $formatted : (string) __('Unknown', Constants::TEXT_DOMAIN);
	}

	/**
	 * Get the size of autoloaded options.
	 *
	 * @return string Formatted autoloaded options size.
	 */
	private function getAutoloadedOptionsSize(): string
	{
		global $wpdb;

		$result = $wpdb->get_var(
			"SELECT SUM(LENGTH(option_value)) FROM {$wpdb->options} WHERE autoload = 'yes'"
		);

		if (null === $result) {
			return (string) __('Unknown', Constants::TEXT_DOMAIN);
		}

		$formatted = size_format((int) $result);

		return $formatted !== false ? $formatted : (string) __('Unknown', Constants::TEXT_DOMAIN);
	}

	/**
	 * Get total number of transients in the options table.
	 *
	 * @return int Number of transients.
	 */
	private function getTransientCount(): int
	{
		global $wpdb;

		$result = $wpdb->get_var(
			"SELECT COUNT(*) FROM {$wpdb->options} WHERE option_name LIKE '\_transient\_%'"
		);

		return (int) $result;
	}

	/**
	 * Render inline styles for the status page.
	 */
	private function renderStyles(): void
	{
		?>
		<style>
			.sctk-status-card {
				background: #fff;
				border: 1px solid #c3c4c7;
				border-radius: 2px;
				margin-bottom: 20px;
			}
			.sctk-status-card h2 {
				display: flex;
				align-items: center;
				justify-content: space-between;
				margin: 0;
				padding: 12px 16px;
				border-bottom: 1px solid #f0f0f1;
				font-size: 14px;
				font-weight: 600;
			}
			.sctk-status-card h2 .button {
				font-size: 12px;
				line-height: 1.5;
				padding: 2px 10px;
			}
			.sctk-status-card table {
				border: 0;
				margin: 0;
				border-spacing: 0;
			}
			.sctk-status-card table td {
				padding: 8px 16px;
				border-bottom: 1px solid #f9f9f9;
				vertical-align: top;
			}
			.sctk-status-card table tr:last-child td {
				border-bottom: 0;
			}
			.sctk-status-card table td:first-child {
				color: #646970;
				width: 250px;
				font-weight: 500;
			}
			.sctk-status-icon-good {
				color: #00a32a;
				font-weight: 600;
			}
			.sctk-status-icon-warning {
				color: #dba617;
				font-weight: 600;
			}
			.sctk-status-icon-bad {
				color: #d63638;
				font-weight: 600;
			}
			.sctk-status-override {
				color: #b8642a;
				font-weight: 500;
				font-style: italic;
			}
			.sctk-copy-status.copied {
				color: #00a32a;
				border-color: #00a32a;
			}
			.sctk-delete-inactive,
			.sctk-delete-inactive-themes {
				background: #b8642a !important;
				border-color: #a05724 !important;
				color: #fff !important;
			}
			.sctk-delete-inactive:hover,
			.sctk-delete-inactive-themes:hover {
				background: #a05724 !important;
				border-color: #8a4a1e !important;
				color: #fff !important;
			}
			.sctk-delete-inactive:focus,
			.sctk-delete-inactive-themes:focus {
				box-shadow: 0 0 0 1px #b8642a !important;
			}
			.sctk-delete-inactive:disabled,
			.sctk-delete-inactive-themes:disabled {
				opacity: 0.6;
				cursor: default;
			}
		</style>
		<?php
	}

	/**
	 * Render inline scripts for copy and delete functionality.
	 *
	 * @param string $deleteNonce Nonce for the delete action.
	 */
	private function renderScripts(string $deleteNonce): void
	{
		?>
		<script>
		(function() {
			// Copy to clipboard
			var copyBtn = document.querySelector('.sctk-copy-status');
			if (copyBtn) {
				copyBtn.addEventListener('click', function() {
					var text = '';
					document.querySelectorAll('.sctk-status-card').forEach(function(card) {
						var title = card.querySelector('h2 span');
						if (title) {
							text += '### ' + title.textContent.trim() + '\n';
						}
						card.querySelectorAll('table tr').forEach(function(row) {
							var cells = row.querySelectorAll('td');
							if (cells.length === 2) {
								text += cells[0].textContent.trim() + ': ' + cells[1].textContent.trim() + '\n';
							}
						});
						text += '\n';
					});

					navigator.clipboard.writeText(text.trim()).then(function() {
						var original = copyBtn.textContent;
						copyBtn.textContent = '<?php echo esc_js(__('Copied!', Constants::TEXT_DOMAIN)); ?>';
						copyBtn.classList.add('copied');
						setTimeout(function() {
							copyBtn.textContent = original;
							copyBtn.classList.remove('copied');
						}, 2000);
					});
				});
			}

			// Delete inactive plugins
			var deleteBtn = document.querySelector('.sctk-delete-inactive');
			if (deleteBtn) {
				deleteBtn.addEventListener('click', function() {
					if (!confirm('<?php echo esc_js(__('Are you sure you want to delete all inactive plugins? This cannot be undone.', Constants::TEXT_DOMAIN)); ?>')) {
						return;
					}

					var button = this;
					var originalText = button.textContent;
					button.disabled = true;
					button.textContent = '<?php echo esc_js(__('Deleting...', Constants::TEXT_DOMAIN)); ?>';

					var data = new FormData();
					data.append('action', 'sctk_delete_inactive_plugins');
					data.append('_wpnonce', button.dataset.nonce);

					fetch(ajaxurl, {
						method: 'POST',
						body: data,
						credentials: 'same-origin'
					})
					.then(function(response) { return response.json(); })
					.then(function(result) {
						if (result.success) {
							var card = button.closest('.sctk-status-card');
							var title = card.querySelector('h2 span');
							title.textContent = '<?php echo esc_js(__('Inactive Plugins (0)', Constants::TEXT_DOMAIN)); ?>';
							card.querySelector('tbody').innerHTML = '<tr><td colspan="2"><?php echo esc_js(__('None', Constants::TEXT_DOMAIN)); ?></td></tr>';
							button.remove();
						} else {
							button.textContent = '<?php echo esc_js(__('Error', Constants::TEXT_DOMAIN)); ?>';
							setTimeout(function() {
								button.textContent = originalText;
								button.disabled = false;
							}, 2000);
						}
					})
					.catch(function() {
						button.textContent = '<?php echo esc_js(__('Error', Constants::TEXT_DOMAIN)); ?>';
						setTimeout(function() {
							button.textContent = originalText;
							button.disabled = false;
						}, 2000);
					});
				});
			}
			// Delete inactive themes
			var deleteThemesBtn = document.querySelector('.sctk-delete-inactive-themes');
			if (deleteThemesBtn) {
				deleteThemesBtn.addEventListener('click', function() {
					if (!confirm('<?php echo esc_js(__('Are you sure you want to delete all inactive themes? This cannot be undone.', Constants::TEXT_DOMAIN)); ?>')) {
						return;
					}

					var button = this;
					var originalText = button.textContent;
					button.disabled = true;
					button.textContent = '<?php echo esc_js(__('Deleting...', Constants::TEXT_DOMAIN)); ?>';

					var data = new FormData();
					data.append('action', 'sctk_delete_inactive_themes');
					data.append('_wpnonce', button.dataset.nonce);

					fetch(ajaxurl, {
						method: 'POST',
						body: data,
						credentials: 'same-origin'
					})
					.then(function(response) { return response.json(); })
					.then(function(result) {
						if (result.success) {
							var card = button.closest('.sctk-status-card');
							var title = card.querySelector('h2 span');
							title.textContent = '<?php echo esc_js(__('Inactive Themes (0)', Constants::TEXT_DOMAIN)); ?>';
							card.querySelector('tbody').innerHTML = '<tr><td colspan="2"><?php echo esc_js(__('None', Constants::TEXT_DOMAIN)); ?></td></tr>';
							button.remove();
						} else {
							button.textContent = '<?php echo esc_js(__('Error', Constants::TEXT_DOMAIN)); ?>';
							setTimeout(function() {
								button.textContent = originalText;
								button.disabled = false;
							}, 2000);
						}
					})
					.catch(function() {
						button.textContent = '<?php echo esc_js(__('Error', Constants::TEXT_DOMAIN)); ?>';
						setTimeout(function() {
							button.textContent = originalText;
							button.disabled = false;
						}, 2000);
					});
				});
			}
		})();
		</script>
		<?php
	}
}
