<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features\DevTools;

use SignocoreToolkit\Application\Constants;
use SignocoreToolkit\Infrastructure\Traits\Singleton;
use SignocoreToolkit\Infrastructure\Traits\ZipDownload;

/**
 * Plugin download functionality.
 *
 * @package SignocoreToolkit\Features\DevTools
 * @since 1.0.0
 */
final class PluginDownload
{
	use Singleton;
	use ZipDownload;

	/**
	 * Link color for download actions.
	 */
	private string $color = '#546e7a';

	/**
	 * Initialize plugin download features.
	 *
	 * Sets up action links and download handler for plugins.
	 */
	protected function init(): void
	{
		add_filter('plugin_action_links', [$this, 'addDownloadLinks'], 10, 2);
		add_action('admin_init', [$this, 'handleDownloadRequest']);
	}

	/**
	 * Add download link to plugin action links.
	 *
	 * Adds a "Download" link for all plugins.
	 *
	 * @param array<string, string> $actions    Existing action links.
	 * @param string                $pluginFile Main plugin file path.
	 * @return array<string, string> Modified action links.
	 */
	public function addDownloadLinks(
		array $actions,
		string $pluginFile
	): array {
		if (!current_user_can('manage_options')) {
			return $actions;
		}

		$url = wp_nonce_url(
			admin_url('plugins.php?action=signocore_download_plugin&plugin=' . urlencode($pluginFile)),
			'signocore_download_plugin_' . $pluginFile
		);

		$icon = '<svg xmlns="http://www.w3.org/2000/svg" width="14" height="13" viewBox="0 0 256 240" fill="currentColor" style="vertical-align: -2px;"><g transform="translate(0,240) scale(0.1,-0.1)" stroke="none"><path d="M1210 2378c-83-43-80-16-80-655 0-342-4-563-9-563-5 0-94 90-196 199-202 216-232 238-307 226-70-12-128-82-128-153 0-16 10-47 22-70 29-51 660-728 699-747 39-21 108-19 146 4 39 25 672 700 696 744 48 89-12 207-115 223-67 11-108-20-308-233l-185-197-5 564c-5 544-6 566-25 600-22 38-93 80-135 80-14 0-46-10-70-22z"/><path d="M91 882c-43-21-70-57-81-108-13-54-12-344 1-423 29-179 161-311 340-340 88-15 1770-15 1859 0 176 28 308 155 340 328 5 29 10 138 10 240 0 174-2 190-22 231-42 83-130 113-208 71-68-36-74-60-80-298l-5-213-28-27-27-28-881-3c-600-2-893 1-919 8-70 19-74 35-80 276-5 195-7 215-26 241-26 36-79 63-124 63-19 0-50-8-69-18z"/></g></svg>';

		$actions['download'] = sprintf(
			'<a href="%s" style="color: %s; font-weight: 600;" aria-label="%s">%s %s</a>',
			esc_url($url),
			esc_attr($this->color),
			esc_attr__('Download Plugin', Constants::TEXT_DOMAIN),
			$icon,
			esc_html__('Download', Constants::TEXT_DOMAIN)
		);

		return $actions;
	}

	/**
	 * Handle plugin download request.
	 *
	 * Processes download requests, validates permissions and nonces,
	 * then generates and streams a zip file of the requested plugin.
	 */
	public function handleDownloadRequest(): void
	{
		if (!$this->isValidDownloadRequest()) {
			return;
		}

		$pluginFile = sanitize_text_field(wp_unslash($_GET['plugin']));

		check_admin_referer('signocore_download_plugin_' . $pluginFile);

		$pluginDir = WP_PLUGIN_DIR . '/' . dirname($pluginFile);
		if (!is_dir($pluginDir)) {
			wp_die(
				esc_html__('Invalid plugin directory.', Constants::TEXT_DOMAIN),
				esc_html__('Plugin Download Error', Constants::TEXT_DOMAIN),
				['response' => 404]
			);
		}

		$pluginVersion = $this->getPluginVersion($pluginDir, $pluginFile);
		$zipName = basename($pluginDir) . '-' . $pluginVersion . '.zip';

		$this->createAndStreamZip($pluginDir, $zipName);
	}

	/**
	 * Validate download request parameters.
	 *
	 * @return bool True if request is valid.
	 */
	private function isValidDownloadRequest(): bool
	{
		return isset($_GET['action'], $_GET['plugin'])
			&& 'signocore_download_plugin' === $_GET['action']
			&& current_user_can('manage_options');
	}

	/**
	 * Get plugin version from plugin data.
	 *
	 * @param string $pluginDir  Plugin directory path.
	 * @param string $pluginFile Plugin file path.
	 * @return string Plugin version number.
	 */
	private function getPluginVersion(string $pluginDir, string $pluginFile): string
	{
		if (!function_exists('get_plugin_data')) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$pluginData = get_plugin_data($pluginDir . '/' . basename($pluginFile));
		return $pluginData['Version'];
	}
}
