<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features\DevTools;

use SignocoreToolkit\Application\Constants;
use SignocoreToolkit\Features\DevTools\Traits\HasDevToolsTabs;
use SignocoreToolkit\Infrastructure\Traits\Singleton;

/**
 * Mail log feature for intercepting and displaying emails.
 *
 * @package SignocoreToolkit\Features\DevTools
 * @since 3.0.0
 */
final class MailCatcher
{
	use HasDevToolsTabs;
	use Singleton;

	/**
	 * Database table name for caught emails.
	 */
	private string $table;

	/**
	 * Number of emails to display per page.
	 */
	private int $perPage = 20;

	/**
	 * Flag to block the next outgoing HTTP request (used to prevent SMTP API calls).
	 */
	private bool $blockNextHttpRequest = false;

	/**
	 * Whether the current email was already captured by preInterceptMail.
	 */
	private bool $emailAlreadyCaptured = false;

	/**
	 * Initialize mail log features.
	 *
	 * Sets up the database table name, mail interception filters,
	 * bulk action handlers, and AJAX preview endpoint.
	 *
	 * Uses two mail filters for full compatibility:
	 * - pre_wp_mail: blocks sending in native WordPress (fires before wp_mail filter)
	 * - wp_mail: captures emails universally (also works with SMTP plugins)
	 */
	protected function init(): void
	{
		global $wpdb;

		$this->table = $wpdb->prefix . 'sctk_caught_emails';

		add_filter('pre_wp_mail', [$this, 'preInterceptMail'], PHP_INT_MAX, 2);
		add_filter('wp_mail', [$this, 'interceptMail'], PHP_INT_MAX, 1);
		add_filter('pre_http_request', [$this, 'blockMailHttpRequest'], PHP_INT_MAX, 3);
		add_action('admin_init', [$this, 'handleBulkAction']);
		add_action('admin_init', [$this, 'handleDeleteAll']);
		add_action('wp_ajax_sctk_email_preview', [$this, 'ajaxGetEmailPreview']);
	}

	/**
	 * Pre-intercept emails in native WordPress.
	 *
	 * Fires only in WordPress core's wp_mail() (not in SMTP plugin replacements).
	 * When blocking is enabled, captures the email and returns false to prevent
	 * sending, which also prevents the wp_mail filter from firing.
	 *
	 * @param null|bool             $return Current pre_wp_mail filter value.
	 * @param array<string, mixed>  $atts   Email attributes.
	 * @return null|bool Null to continue sending, false to block.
	 */
	public function preInterceptMail(null|bool $return, array $atts): null|bool
	{
		if (!$this->shouldPreventSending()) {
			return $return;
		}

		$this->storeEmail($atts, 'blocked');
		$this->emailAlreadyCaptured = true;

		return false;
	}

	/**
	 * Intercept outgoing WordPress emails.
	 *
	 * Captures email data and stores it in the database. Uses the wp_mail filter
	 * for compatibility with SMTP plugins that replace the pluggable wp_mail() function.
	 * Skips capturing if already handled by preInterceptMail.
	 *
	 * @param array<string, mixed> $atts Email attributes (to, subject, message, headers, attachments).
	 * @return array<string, mixed> The email attributes (unmodified).
	 */
	public function interceptMail(array $atts): array
	{
		if ($this->emailAlreadyCaptured) {
			$this->emailAlreadyCaptured = false;

			return $atts;
		}

		$prevented = $this->shouldPreventSending();
		$status = $prevented ? 'blocked' : 'sent';

		if ($this->shouldLogEmail($status)) {
			$this->storeEmail($atts, $status);
		}

		if ($prevented) {
			$this->blockNextHttpRequest = true;
		}

		return $atts;
	}

	/**
	 * Store an intercepted email in the database.
	 *
	 * @param array<string, mixed> $atts   Email attributes.
	 * @param string               $status Email status ('sent', 'blocked').
	 */
	private function storeEmail(array $atts, string $status): void
	{
		global $wpdb;

		$to = $atts['to'] ?? '';
		if (is_array($to)) {
			$to = implode(', ', $to);
		}
		$to = (string) $to;

		$subject = (string) ($atts['subject'] ?? '');
		$body = (string) ($atts['message'] ?? '');

		$headers = $atts['headers'] ?? '';
		if (is_array($headers)) {
			$headers = implode("\n", $headers);
		}
		$headers = (string) $headers;

		$attachments = $atts['attachments'] ?? [];
		$attachmentsValue = null;
		if (is_array($attachments) && [] !== $attachments) {
			$attachmentsValue = maybe_serialize($attachments);
		}

		$wpdb->insert(
			$this->table,
			[
				'to_address'  => $to,
				'subject'     => $subject,
				'body'        => $body,
				'headers'     => $headers,
				'attachments' => $attachmentsValue,
				'status'      => $status,
				'created_at'  => current_time('mysql'),
			],
			['%s', '%s', '%s', '%s', '%s', '%s', '%s']
		);
	}

	/**
	 * Block outgoing HTTP requests when mail sending is prevented.
	 *
	 * Short-circuits the next HTTP POST request after interceptMail sets
	 * the blocking flag, returning a fake success response so SMTP plugins
	 * don't trigger error handling for failed API calls.
	 *
	 * @param false|array<string, mixed>|\WP_Error $preempt Whether to preempt the request.
	 * @param array<string, mixed>                 $args    HTTP request arguments.
	 * @param string                               $url     The request URL.
	 * @return false|array<string, mixed>|\WP_Error Fake response when blocking, pass-through otherwise.
	 */
	public function blockMailHttpRequest(false|array|\WP_Error $preempt, array $args, string $url): false|array|\WP_Error
	{
		if (!$this->blockNextHttpRequest) {
			return $preempt;
		}

		if ('POST' !== strtoupper($args['method'] ?? '')) {
			return $preempt;
		}

		$this->blockNextHttpRequest = false;

		return [
			'response' => ['code' => 200, 'message' => 'OK'],
			'body'     => wp_json_encode(['message' => 'Blocked by Signocore Toolkit']),
			'headers'  => [],
			'cookies'  => [],
		];
	}

	/**
	 * Render the mail log admin page.
	 *
	 * Displays an inbox-style UI with search, bulk actions, email table,
	 * pagination, and an AJAX-powered preview panel.
	 */
	public function renderPage(): void
	{
		$search = isset($_GET['s']) ? sanitize_text_field(wp_unslash($_GET['s'])) : '';
		$currentPage = isset($_GET['paged']) ? absint($_GET['paged']) : 1;
		if (0 === $currentPage) {
			$currentPage = 1;
		}

		$totalEmails = $this->getTotalCount($search);
		$emails = $this->getEmails($currentPage, $search);
		$previewNonce = wp_create_nonce('sctk_preview_nonce');
		$pageSlug = isset($_GET['page']) ? sanitize_text_field(wp_unslash($_GET['page'])) : '';

		$paginationLinks = paginate_links([
			'base'      => add_query_arg('paged', '%#%'),
			'format'    => '',
			'current'   => $currentPage,
			'total'     => (int) ceil($totalEmails / $this->perPage),
			'prev_text' => '&laquo;',
			'next_text' => '&raquo;',
		]);

		?>
		<div class="wrap">
			<h1 class="wp-heading-inline">
				<?php echo esc_html__('Mail Log', Constants::TEXT_DOMAIN); ?>
				<span class="sctk-count-badge"><?php echo esc_html((string) $totalEmails); ?></span>
			</h1>
			<hr class="wp-header-end">

			<?php
			$this->renderMainTabs();
			$this->renderDevToolsTabs('signocore-toolkit-mail-log');
			?>

			<?php if (isset($_GET['deleted'])) : ?>
				<div class="notice notice-success is-dismissible">
					<p>
						<?php
						printf(
							/* translators: %d: number of deleted emails */
							esc_html__('%d email(s) deleted.', Constants::TEXT_DOMAIN),
							absint($_GET['deleted'])
						);
						?>
					</p>
				</div>
			<?php endif; ?>

			<?php if (isset($_GET['deleted_all'])) : ?>
				<div class="notice notice-success is-dismissible">
					<p><?php echo esc_html__('Log emptied.', Constants::TEXT_DOMAIN); ?></p>
				</div>
			<?php endif; ?>

			<style>
				.sctk-count-badge {
					display: inline-block;
					min-width: 22px;
					padding: 2px 8px;
					font-size: 13px;
					font-weight: 600;
					line-height: 1.4;
					text-align: center;
					background: #2271b1;
					color: #fff;
					border-radius: 10px;
					vertical-align: middle;
					margin-left: 4px;
				}
				.sctk-search-box {
					float: right;
					margin: 0 0 10px;
				}
				.sctk-search-box input[type="search"] {
					width: 280px;
				}
				.sctk-status-badge {
					display: inline-block;
					padding: 2px 8px;
					font-size: 12px;
					font-weight: 600;
					line-height: 1.5;
					border-radius: 3px;
					text-transform: uppercase;
					letter-spacing: 0.3px;
				}
				.sctk-status-sent {
					background: #d4edda;
					color: #155724;
				}
				.sctk-status-blocked {
					background: #fff3cd;
					color: #856404;
				}
				.sctk-status-caught {
					background: #e2e3e5;
					color: #383d41;
				}
				.sctk-subject-link {
					cursor: pointer;
					color: #2271b1;
					text-decoration: none;
					font-weight: 600;
				}
				.sctk-subject-link:hover {
					color: #135e96;
					text-decoration: underline;
				}
				.sctk-modal-overlay {
					display: none;
					position: fixed;
					inset: 0;
					background: rgba(0, 0, 0, 0.6);
					z-index: 100000;
				}
				.sctk-modal {
					position: fixed;
					top: 50%;
					left: 50%;
					transform: translate(-50%, -50%);
					width: 90%;
					max-width: 800px;
					max-height: 85vh;
					background: #fff;
					border-radius: 4px;
					box-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
					display: flex;
					flex-direction: column;
					z-index: 100001;
				}
				.sctk-modal-header {
					display: flex;
					justify-content: space-between;
					align-items: flex-start;
					padding: 16px 20px;
					border-bottom: 1px solid #ddd;
				}
				.sctk-modal-header h3 {
					margin: 0;
					font-size: 16px;
					line-height: 1.4;
					flex: 1;
					padding-right: 12px;
				}
				.sctk-modal-close {
					background: none;
					border: none;
					cursor: pointer;
					font-size: 20px;
					color: #666;
					padding: 0;
					line-height: 1;
					flex-shrink: 0;
				}
				.sctk-modal-close:hover {
					color: #d63638;
				}
				.sctk-modal-meta {
					padding: 12px 20px;
					border-bottom: 1px solid #eee;
					color: #666;
					font-size: 13px;
				}
				.sctk-modal-meta strong {
					color: #333;
				}
				.sctk-modal-meta > div {
					margin-bottom: 4px;
				}
				.sctk-modal-meta > div:last-child {
					margin-bottom: 0;
				}
				.sctk-modal-body {
					padding: 20px;
					overflow-y: auto;
					flex: 1;
				}
				.sctk-bulk-actions {
					display: flex;
					align-items: center;
					gap: 8px;
					margin: 10px 0;
				}
				.sctk-delete-all {
					float: right;
				}
				.sctk-pagination {
					margin: 10px 0;
					text-align: right;
				}
				.sctk-pagination .page-numbers {
					padding: 4px 10px;
					background: #f0f0f1;
					border: 1px solid #ddd;
					text-decoration: none;
					color: #2271b1;
				}
				.sctk-pagination .page-numbers.current {
					background: #2271b1;
					color: #fff;
					border-color: #2271b1;
				}
				.sctk-table-footer {
					display: flex;
					justify-content: space-between;
					align-items: center;
					margin-top: 10px;
				}
			</style>

			<?php if (0 < $totalEmails) : ?>
				<form method="post" class="sctk-delete-all" style="float:right;margin:0 0 10px 6px;">
					<?php wp_nonce_field('sctk_mail_delete_all', 'sctk_mail_delete_all'); ?>
					<input
						type="submit"
						class="button button-link-delete"
						value="<?php echo esc_attr__('Empty Log', Constants::TEXT_DOMAIN); ?>"
						onclick="return confirm('<?php echo esc_js(__('Are you sure you want to empty the log?', Constants::TEXT_DOMAIN)); ?>');"
					>
				</form>
			<?php endif; ?>

			<form method="get" class="sctk-search-box">
				<input type="hidden" name="page" value="<?php echo esc_attr($pageSlug); ?>">
				<label class="screen-reader-text" for="sctk-email-search">
					<?php echo esc_html__('Search emails', Constants::TEXT_DOMAIN); ?>
				</label>
				<input
					type="search"
					id="sctk-email-search"
					name="s"
					value="<?php echo esc_attr($search); ?>"
					placeholder="<?php echo esc_attr__('Search by subject or recipient...', Constants::TEXT_DOMAIN); ?>"
				>
				<input
					type="submit"
					class="button"
					value="<?php echo esc_attr__('Search', Constants::TEXT_DOMAIN); ?>"
				>
			</form>

			<form method="post" id="sctk-bulk-form">
				<?php wp_nonce_field('sctk_mail_bulk_action', 'sctk_mail_bulk_action'); ?>

				<table class="widefat striped">
					<thead>
						<tr>
							<td class="manage-column check-column">
								<input type="checkbox" id="sctk-select-all">
							</td>
							<th><?php echo esc_html__('Subject', Constants::TEXT_DOMAIN); ?></th>
							<th><?php echo esc_html__('To', Constants::TEXT_DOMAIN); ?></th>
							<th><?php echo esc_html__('Status', Constants::TEXT_DOMAIN); ?></th>
							<th><?php echo esc_html__('Date', Constants::TEXT_DOMAIN); ?></th>
						</tr>
					</thead>
					<tbody>
						<?php if ([] === $emails) : ?>
							<tr>
								<td colspan="5">
									<?php echo esc_html__('No emails found.', Constants::TEXT_DOMAIN); ?>
								</td>
							</tr>
						<?php else : ?>
							<?php foreach ($emails as $email) : ?>
								<tr>
									<th class="check-column">
										<input
											type="checkbox"
											name="email_ids[]"
											value="<?php echo esc_attr((string) $email->id); ?>"
											class="sctk-email-checkbox"
										>
									</th>
									<td>
										<a
											href="#sctk-email-preview"
											class="sctk-subject-link"
											data-email-id="<?php echo esc_attr((string) $email->id); ?>"
										>
											<?php echo esc_html($email->subject ?: __('(no subject)', Constants::TEXT_DOMAIN)); ?>
										</a>
									</td>
									<td><?php echo esc_html($email->to_address); ?></td>
									<td>
										<?php
										$statusClass = 'sctk-status-caught';
										if ('sent' === $email->status) {
											$statusClass = 'sctk-status-sent';
										} elseif ('blocked' === $email->status) {
											$statusClass = 'sctk-status-blocked';
										}
										?>
										<span class="sctk-status-badge <?php echo esc_attr($statusClass); ?>">
											<?php echo esc_html($email->status); ?>
										</span>
									</td>
									<td title="<?php echo esc_attr($email->created_at); ?>">
										<?php
										$timestamp = strtotime($email->created_at);
										if ($timestamp !== false) {
											printf(
												/* translators: %s: human-readable time difference */
												esc_html__('%s ago', Constants::TEXT_DOMAIN),
												esc_html(human_time_diff($timestamp, current_time('timestamp')))
											);
										} else {
											echo esc_html($email->created_at);
										}
										?>
									</td>
								</tr>
							<?php endforeach; ?>
						<?php endif; ?>
					</tbody>
				</table>

				<div class="sctk-table-footer">
					<div class="sctk-bulk-actions">
						<select name="bulk_action">
							<option value=""><?php echo esc_html__('Bulk Actions', Constants::TEXT_DOMAIN); ?></option>
							<option value="delete"><?php echo esc_html__('Delete Selected', Constants::TEXT_DOMAIN); ?></option>
						</select>
						<input
							type="submit"
							class="button action"
							value="<?php echo esc_attr__('Apply', Constants::TEXT_DOMAIN); ?>"
						>
					</div>

					<?php if ($paginationLinks) : ?>
						<div class="sctk-pagination">
							<?php echo wp_kses_post($paginationLinks); ?>
						</div>
					<?php endif; ?>
				</div>
			</form>

			<div id="sctk-modal-overlay" class="sctk-modal-overlay">
				<div class="sctk-modal">
					<div class="sctk-modal-header">
						<h3 id="sctk-preview-subject"></h3>
						<button type="button" class="sctk-modal-close" aria-label="<?php echo esc_attr__('Close', Constants::TEXT_DOMAIN); ?>">&times;</button>
					</div>
					<div class="sctk-modal-meta">
						<div><strong><?php echo esc_html__('To:', Constants::TEXT_DOMAIN); ?></strong> <span id="sctk-preview-to"></span></div>
						<div><strong><?php echo esc_html__('Date:', Constants::TEXT_DOMAIN); ?></strong> <span id="sctk-preview-date"></span></div>
						<div><strong><?php echo esc_html__('Status:', Constants::TEXT_DOMAIN); ?></strong> <span id="sctk-preview-status"></span></div>
						<div><strong><?php echo esc_html__('Headers:', Constants::TEXT_DOMAIN); ?></strong> <pre id="sctk-preview-headers" style="margin:5px 0;white-space:pre-wrap;font-size:12px;color:#666;"></pre></div>
					</div>
					<div class="sctk-modal-body" id="sctk-preview-body"></div>
				</div>
			</div>

			<script>
			jQuery(function($) {
				var previewNonce = '<?php echo esc_js($previewNonce); ?>';
				var $overlay = $('#sctk-modal-overlay');

				function closeModal() {
					$overlay.hide();
					$('#sctk-preview-body').html('');
				}

				// Select all checkbox
				$('#sctk-select-all').on('change', function() {
					$('.sctk-email-checkbox').prop('checked', $(this).prop('checked'));
				});

				// Close modal
				$('.sctk-modal-close').on('click', closeModal);
				$overlay.on('click', function(e) {
					if (e.target === this) {
						closeModal();
					}
				});
				$(document).on('keydown', function(e) {
					if (27 === e.keyCode && $overlay.is(':visible')) {
						closeModal();
					}
				});

				// Email preview via AJAX
				$('.sctk-subject-link').on('click', function(e) {
					e.preventDefault();
					var emailId = $(this).data('email-id');

					$overlay.show();
					$('#sctk-preview-subject').text('<?php echo esc_js(__('Loading...', Constants::TEXT_DOMAIN)); ?>');
					$('#sctk-preview-to').text('');
					$('#sctk-preview-date').text('');
					$('#sctk-preview-status').html('');
					$('#sctk-preview-headers').text('');
					$('#sctk-preview-body').html('');

					$.ajax({
						url: ajaxurl,
						method: 'POST',
						data: {
							action: 'sctk_email_preview',
							email_id: emailId,
							_ajax_nonce: previewNonce
						},
						success: function(response) {
							if (response.success) {
								var data = response.data;
								$('#sctk-preview-subject').text(data.subject || '<?php echo esc_js(__('(no subject)', Constants::TEXT_DOMAIN)); ?>');
								$('#sctk-preview-to').text(data.to_address);
								$('#sctk-preview-date').text(data.created_at);
								$('#sctk-preview-status').text(data.status);
								$('#sctk-preview-headers').text(data.headers || '-');
								$('#sctk-preview-body').html(
									'<iframe srcdoc="' + $('<div/>').text(data.body).html().replace(/"/g, '&quot;') + '" sandbox="allow-same-origin" style="width:100%;height:400px;border:none;background:#fff;"></iframe>'
								);
							} else {
								$('#sctk-preview-subject').text('<?php echo esc_js(__('Error loading email preview.', Constants::TEXT_DOMAIN)); ?>');
							}
						},
						error: function() {
							$('#sctk-preview-subject').text('<?php echo esc_js(__('Error loading email preview.', Constants::TEXT_DOMAIN)); ?>');
						}
					});
				});
			});
			</script>
		</div>
		<?php
	}

	/**
	 * Handle bulk delete action for selected emails.
	 *
	 * Verifies nonce and capability, then deletes selected email records
	 * and redirects back with a success count.
	 */
	public function handleBulkAction(): void
	{
		if (!isset($_POST['sctk_mail_bulk_action'])) {
			return;
		}

		if (!check_admin_referer('sctk_mail_bulk_action', 'sctk_mail_bulk_action')) {
			return;
		}

		if (!current_user_can('manage_options')) {
			wp_die(esc_html__('You do not have permission to perform this action.', Constants::TEXT_DOMAIN));
		}

		$bulkAction = isset($_POST['bulk_action']) ? sanitize_text_field(wp_unslash($_POST['bulk_action'])) : '';
		if ('delete' !== $bulkAction) {
			return;
		}

		$emailIds = isset($_POST['email_ids']) ? array_map('absint', (array) $_POST['email_ids']) : [];
		if ([] === $emailIds) {
			return;
		}

		global $wpdb;

		$deleted = 0;
		foreach ($emailIds as $id) {
			if (0 < $id) {
				$wpdb->delete($this->table, ['id' => $id], ['%d']);
				$deleted++;
			}
		}

		$redirectUrl = add_query_arg(
			'deleted',
			$deleted,
			wp_get_referer() ?: admin_url('admin.php')
		);

		wp_safe_redirect($redirectUrl);
		exit;
	}

	/**
	 * Handle delete all emails action.
	 *
	 * Verifies nonce and capability, truncates the email table,
	 * and redirects back with success notice.
	 */
	public function handleDeleteAll(): void
	{
		if (!isset($_POST['sctk_mail_delete_all'])) {
			return;
		}

		if (!check_admin_referer('sctk_mail_delete_all', 'sctk_mail_delete_all')) {
			return;
		}

		if (!current_user_can('manage_options')) {
			wp_die(esc_html__('You do not have permission to perform this action.', Constants::TEXT_DOMAIN));
		}

		global $wpdb;

		// Table name is safe - constructed from $wpdb->prefix in init()
		$wpdb->query("TRUNCATE TABLE {$this->table}");

		$redirectUrl = add_query_arg(
			'deleted_all',
			'1',
			wp_get_referer() ?: admin_url('admin.php?page=signocore-toolkit-mail-log')
		);

		wp_safe_redirect($redirectUrl);
		exit;
	}

	/**
	 * Handle AJAX request for email preview.
	 *
	 * Returns email data as JSON for the preview panel.
	 * Requires valid nonce and manage_options capability.
	 */
	public function ajaxGetEmailPreview(): void
	{
		check_ajax_referer('sctk_preview_nonce');

		if (!current_user_can('manage_options')) {
			wp_send_json_error(['message' => __('Insufficient permissions.', Constants::TEXT_DOMAIN)]);
		}

		$emailId = 0;
		if (isset($_POST['email_id'])) {
			$emailId = absint($_POST['email_id']);
		} elseif (isset($_GET['email_id'])) {
			$emailId = absint($_GET['email_id']);
		}

		if (0 === $emailId) {
			wp_send_json_error(['message' => __('Invalid email ID.', Constants::TEXT_DOMAIN)]);
		}

		global $wpdb;

		$email = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT subject, to_address, body, headers, created_at, status FROM {$this->table} WHERE id = %d",
				$emailId
			)
		);

		if (null === $email) {
			wp_send_json_error(['message' => __('Email not found.', Constants::TEXT_DOMAIN)]);
		}

		wp_send_json_success([
			'subject'    => $email->subject,
			'to_address' => $email->to_address,
			'body'       => $email->body,
			'headers'    => $email->headers,
			'created_at' => $email->created_at,
			'status'     => $email->status,
		]);
	}

	/**
	 * Get paginated emails from the database.
	 *
	 * @param int    $page   Current page number (1-based).
	 * @param string $search Search term for subject or recipient filtering.
	 * @return array<int, object{id: int, subject: string, to_address: string, status: string, created_at: string, body: string, headers: string}> Array of email row objects.
	 */
	private function getEmails(int $page, string $search): array
	{
		global $wpdb;

		$offset = ($page - 1) * $this->perPage;

		if ('' !== $search) {
			$like = '%' . $wpdb->esc_like($search) . '%';

			return $wpdb->get_results(
				$wpdb->prepare(
					"SELECT * FROM {$this->table} WHERE subject LIKE %s OR to_address LIKE %s ORDER BY created_at DESC LIMIT %d OFFSET %d",
					$like,
					$like,
					$this->perPage,
					$offset
				)
			);
		}

		return $wpdb->get_results(
			$wpdb->prepare(
				"SELECT * FROM {$this->table} ORDER BY created_at DESC LIMIT %d OFFSET %d",
				$this->perPage,
				$offset
			)
		);
	}

	/**
	 * Get total count of emails matching search criteria.
	 *
	 * @param string $search Search term for subject or recipient filtering.
	 * @return int Total number of matching emails.
	 */
	private function getTotalCount(string $search): int
	{
		global $wpdb;

		if ('' !== $search) {
			$like = '%' . $wpdb->esc_like($search) . '%';

			return (int) $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM {$this->table} WHERE subject LIKE %s OR to_address LIKE %s",
					$like,
					$like
				)
			);
		}

		return (int) $wpdb->get_var("SELECT COUNT(*) FROM {$this->table}");
	}

	/**
	 * Determine if an email should be logged based on the log mode setting.
	 *
	 * @param string $status The email status ('sent' or 'blocked').
	 * @return bool True if the email should be stored.
	 */
	private function shouldLogEmail(string $status): bool
	{
		$logMode = get_option('sctk_mail_log_mode', 'all');

		if ('blocked' === $logMode) {
			return 'blocked' === $status;
		}

		return true;
	}

	/**
	 * Determine if email sending should be prevented.
	 *
	 * Checks whether the prevent-sending option is enabled and
	 * the environment is not production. Uses EnvironmentBanner's
	 * detection which includes domain-based fallback.
	 *
	 * @return bool True if sending should be blocked.
	 */
	private function shouldPreventSending(): bool
	{
		$preventOption = get_option('sctk_mail_prevent_sending', false);

		if (!$preventOption) {
			return false;
		}

		$environment = EnvironmentBanner::load()->getEnvironment();

		return 'production' !== $environment;
	}
}
