<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features\DevTools;

use SignocoreToolkit\Application\Constants;
use SignocoreToolkit\Infrastructure\Traits\Singleton;

/**
 * Environment banner for the admin bar.
 *
 * @package SignocoreToolkit\Features\DevTools
 * @since 3.0.0
 */
final class EnvironmentBanner
{
	use Singleton;

	/**
	 * Detected environment name.
	 */
	private string $environment;

	/**
	 * Color mapping for each environment.
	 *
	 * @var array<string, string>
	 */
	private const COLORS = [
		'development' => '#4CAF50',
		'staging' => '#FF9800',
		'production' => '#f44336',
	];

	/**
	 * Display labels for each environment.
	 *
	 * @var array<string, string>
	 */
	private const LABELS = [
		'development' => 'Development',
		'staging' => 'Staging',
		'production' => 'Production',
	];

	/**
	 * Initialize environment banner features.
	 *
	 * Detects the current environment and registers hooks for
	 * rendering the admin bar indicator and styles.
	 */
	protected function init(): void
	{
		$this->environment = $this->detectEnvironment();

		add_action('admin_bar_menu', [$this, 'addBannerToAdminBar'], 5);
		add_action('admin_head', [$this, 'renderStyles']);
		add_action('wp_head', [$this, 'renderStyles']);
	}

	/**
	 * Get the detected environment name.
	 *
	 * @return string The detected environment ('development', 'staging', or 'production').
	 */
	public function getEnvironment(): string
	{
		return $this->environment;
	}

	/**
	 * Add environment indicator node to the admin bar.
	 *
	 * @param \WP_Admin_Bar $adminBar WordPress admin bar instance.
	 */
	public function addBannerToAdminBar(\WP_Admin_Bar $adminBar): void
	{
		if (!is_admin_bar_showing()) {
			return;
		}

		$adminBar->add_node([
			'id' => 'sctk-environment',
			'title' => self::LABELS[$this->environment] ?? ucfirst($this->environment),
			'meta' => ['class' => 'sctk-env-' . $this->environment],
		]);
	}

	/**
	 * Render inline styles for the environment indicator.
	 *
	 * Colors the environment node background for development and staging.
	 * For production, only adds a small colored dot indicator on the node.
	 */
	public function renderStyles(): void
	{
		if (!is_admin_bar_showing()) {
			return;
		}

		$color = self::COLORS[$this->environment] ?? self::COLORS['production'];

		echo '<style>' . "\n";

		if ('development' === $this->environment || 'staging' === $this->environment) {
			echo '#wpadminbar #wp-admin-bar-sctk-environment > .ab-item {' . "\n";
			echo '    background-color: ' . $color . ' !important;' . "\n";
			echo '    color: #fff !important;' . "\n";
			echo '}' . "\n";
		} else {
			echo '#wpadminbar #wp-admin-bar-sctk-environment .ab-item::before {' . "\n";
			echo '    content: "" !important;' . "\n";
			echo '    display: inline-block !important;' . "\n";
			echo '    width: 8px !important;' . "\n";
			echo '    height: 8px !important;' . "\n";
			echo '    border-radius: 50% !important;' . "\n";
			echo '    background-color: ' . $color . ' !important;' . "\n";
			echo '    margin-right: 6px;' . "\n";
			echo '    vertical-align: middle;' . "\n";
			echo '    padding: 0 !important;' . "\n";
			echo '    top: 12px !important;' . "\n";
			echo '    font: 0/0 a !important;' . "\n";
			echo '}' . "\n";
		}

		echo '#wpadminbar #wp-admin-bar-sctk-environment .ab-item {' . "\n";
		echo '    font-weight: bold;' . "\n";
		echo '    text-transform: uppercase;' . "\n";
		echo '}' . "\n";

		echo '</style>' . "\n";
	}

	/**
	 * Detect the current environment.
	 *
	 * Checks in order: manual override option, WordPress environment type,
	 * and falls back to domain-based detection.
	 *
	 * @return string The detected environment ('development', 'staging', or 'production').
	 */
	private function detectEnvironment(): string
	{
		$override = get_option('sctk_environment_override', 'auto');

		if ('auto' !== $override && '' !== $override) {
			return $override;
		}

		if (function_exists('wp_get_environment_type')) {
			$wpEnv = wp_get_environment_type();

			return match ($wpEnv) {
				'local', 'development' => 'development',
				'staging' => 'staging',
				default => 'production',
			};
		}

		$host = $_SERVER['HTTP_HOST'] ?? '';

		if (str_ends_with($host, '.test') || str_ends_with($host, '.local')) {
			return 'development';
		}

		if (str_starts_with($host, 'staging.')) {
			return 'staging';
		}

		return 'production';
	}
}
