<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features\DevTools;

use SignocoreToolkit\Application\Constants;
use SignocoreToolkit\Infrastructure\Traits\Singleton;

/**
 * Debugging feature management.
 *
 * @package SignocoreToolkit\Features\DevTools
 * @since 1.0.0
 */
final class Debugging
{
	use Singleton;

	/**
     * Initialize debugging features.
     *
     * Sets up admin notices and configures error reporting
     * when WordPress debug mode is active.
     */
    protected function init(): void
	{
		if (defined('WP_DEBUG') && WP_DEBUG) {
			add_action('admin_notices', [$this, 'displayDebugNotice']);
			$this->configureErrorReporting();
		}
	}

	/**
     * Display debug mode warning notice in admin.
     *
     * Shows a dismissible warning when WP_DEBUG is enabled
     * to remind administrators to disable it in production.
     */
    public function displayDebugNotice(): void
	{
		printf(
			'<div class="notice notice-warning is-dismissible"><p>%s</p></div>',
			esc_html__(
				'Signocore Toolkit: Debugging is enabled. Please disable it in production environments.',
				Constants::TEXT_DOMAIN
			)
		);
	}

	/**
     * Configure PHP error reporting for debugging.
     *
     * Enables comprehensive error reporting when debug mode is active.
     * Uses ini_set only if the function is available and not disabled.
     */
    private function configureErrorReporting(): void
	{
		if (!function_exists('ini_set')) {
			return;
		}

		// Enable all error reporting
		error_reporting(E_ALL);

		// Attempt to configure display settings
		ini_set('display_errors', '1');
		ini_set('display_startup_errors', '1');
	}
}
