<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features\DevTools;

use SignocoreToolkit\Infrastructure\Traits\Singleton;

/**
 * Database management for DevTools features.
 *
 * @package SignocoreToolkit\Features\DevTools
 * @since 1.0.0
 */
final class Database
{
	use Singleton;

	/**
	 * Database schema version.
	 *
	 * Increment this when making schema changes.
	 *
	 * @var string
	 */
	private const DB_VERSION = '2.0.0';

	/**
	 * Option name for storing database version.
	 *
	 * @var string
	 */
	private const DB_VERSION_OPTION = 'sctk_dev_db_version';

	/**
	 * Legacy option name for database version (pre-2.x).
	 *
	 * @var string
	 */
	private const LEGACY_DB_VERSION_OPTION = 'swift_dev_db_version';

	/**
     * Initialize database hooks.
     */
    protected function init(): void
	{
		add_action('admin_init', [$this, 'maybeUpdateDatabase'], 0);
	}

	/**
     * Check and update database if needed.
     *
     * Compares stored database version with current version
     * and runs migrations if necessary.
     */
    public function maybeUpdateDatabase(): void
	{
		$this->migrateLegacyOption();

		$currentVersion = get_option(self::DB_VERSION_OPTION, '0.0.0');

		if (version_compare($currentVersion, self::DB_VERSION, '<')) {
			$this->runMigrations($currentVersion);
			update_option(self::DB_VERSION_OPTION, self::DB_VERSION, false);
		}
	}

	/**
	 * Migrate legacy option key to new key.
	 */
	private function migrateLegacyOption(): void
	{
		$legacyVersion = get_option(self::LEGACY_DB_VERSION_OPTION, false);

		if ($legacyVersion !== false) {
			if (get_option(self::DB_VERSION_OPTION, false) === false) {
				update_option(self::DB_VERSION_OPTION, $legacyVersion, false);
			}
			delete_option(self::LEGACY_DB_VERSION_OPTION);
		}
	}

	/**
     * Run database migrations based on version.
     *
     * @param string $fromVersion Current database version.
     */
    private function runMigrations(string $fromVersion): void
	{
		if (version_compare($fromVersion, '2.0.0', '<')) {
			$this->createCaughtEmailsTable();
		}
	}

	/**
	 * Create the caught emails table for the mail catcher feature.
	 *
	 * Stores intercepted emails with subject, body, headers,
	 * and delivery status information.
	 */
	private function createCaughtEmailsTable(): void
	{
		global $wpdb;

		$table = $wpdb->prefix . 'sctk_caught_emails';
		$charsetCollate = $wpdb->get_charset_collate();

		$tableExistsQuery = $wpdb->prepare('SHOW TABLES LIKE %s', $wpdb->esc_like($table));
		if ($wpdb->get_var($tableExistsQuery) === $table) {
			return;
		}

		$sql = "CREATE TABLE `{$table}` (
			`id` bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
			`to_address` varchar(255) NOT NULL DEFAULT '',
			`subject` varchar(255) NOT NULL DEFAULT '',
			`body` longtext NOT NULL,
			`headers` text,
			`attachments` text,
			`status` varchar(20) NOT NULL DEFAULT 'caught',
			`created_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY (`id`),
			KEY `to_address_index` (`to_address`),
			KEY `subject_index` (`subject`),
			KEY `status_index` (`status`),
			KEY `created_at_index` (`created_at`)
		) {$charsetCollate};";

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		dbDelta($sql);
	}
}
