<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features\DevTools;

use SignocoreToolkit\Application\Constants;
use SignocoreToolkit\Features\DevTools\Traits\HasDevToolsTabs;
use SignocoreToolkit\Infrastructure\Traits\Singleton;

/**
 * Cron event viewer and management.
 *
 * @package SignocoreToolkit\Features\DevTools
 * @since 3.0.0
 */
final class CronViewer
{
	use HasDevToolsTabs;
	use Singleton;

	/**
	 * Initialize cron viewer features.
	 *
	 * Registers the handler for manually running cron events.
	 */
	protected function init(): void
	{
		add_action('wp_ajax_sctk_run_cron', [$this, 'handleRunNowAjax']);
	}

	/**
	 * Render the cron viewer admin page.
	 *
	 * Displays a table of all scheduled WP-Cron events with their
	 * hook names, schedules, arguments, and action links.
	 */
	public function renderPage(): void
	{
		$events = $this->getCronEvents();
		$currentTime = time();
		?>
		<div class="wrap">
			<h1><?php echo esc_html__('Cron Jobs', Constants::TEXT_DOMAIN); ?></h1>
			<hr class="wp-header-end">

			<?php
			$this->renderMainTabs();
			$this->renderDevToolsTabs('signocore-toolkit-cron');
			?>

			<style>
				.sctk-cron-overdue {
					background-color: #fcf0f0 !important;
				}
				.sctk-cron-badge-overdue {
					display: inline-block;
					background: #d63638;
					color: #fff;
					font-size: 11px;
					font-weight: 600;
					padding: 2px 6px;
					border-radius: 3px;
					margin-left: 4px;
					vertical-align: middle;
				}
				.sctk-cron-args {
					max-width: 300px;
					overflow-wrap: break-word;
					word-break: break-all;
				}
				.sctk-cron-run {
					background: none;
					border: none;
					color: #2271b1;
					cursor: pointer;
					padding: 0;
					font-size: inherit;
					text-decoration: underline;
				}
				.sctk-cron-run:hover {
					color: #135e96;
				}
				.sctk-cron-run:disabled {
					color: #a7aaad;
					cursor: default;
					text-decoration: none;
				}
				.sctk-cron-run .spinner {
					float: none;
					margin: 0 0 0 4px;
					vertical-align: middle;
				}
				.sctk-cron-done {
					color: #00a32a;
					font-weight: 600;
				}
				.sctk-cron-error {
					color: #d63638;
					font-weight: 600;
				}
			</style>

			<table class="widefat striped">
				<thead>
					<tr>
						<th><?php echo esc_html__('Hook Name', Constants::TEXT_DOMAIN); ?></th>
						<th><?php echo esc_html__('Next Run', Constants::TEXT_DOMAIN); ?></th>
						<th><?php echo esc_html__('Schedule', Constants::TEXT_DOMAIN); ?></th>
						<th><?php echo esc_html__('Arguments', Constants::TEXT_DOMAIN); ?></th>
						<th><?php echo esc_html__('Actions', Constants::TEXT_DOMAIN); ?></th>
					</tr>
				</thead>
				<tbody>
					<?php if ([] === $events) : ?>
						<tr>
							<td colspan="5"><?php echo esc_html__('No cron events found.', Constants::TEXT_DOMAIN); ?></td>
						</tr>
					<?php else : ?>
						<?php foreach ($events as $event) : ?>
							<?php
							$isOverdue = $this->isOverdue($event['timestamp']);
							$timeDiff = human_time_diff($event['timestamp'], $currentTime);
							$timeLabel = $isOverdue
								? sprintf(
									/* translators: %s: human-readable time difference */
									esc_html__('%s ago', Constants::TEXT_DOMAIN),
									$timeDiff
								)
								: sprintf(
									/* translators: %s: human-readable time difference */
									esc_html__('%s from now', Constants::TEXT_DOMAIN),
									$timeDiff
								);
							$datetimeFormatted = wp_date('Y-m-d H:i:s', $event['timestamp']);
							$sig = md5(serialize($event['args']));
							$nonce = wp_create_nonce('sctk_run_cron_' . $event['hook'] . '_' . $sig);
							?>
							<tr<?php echo $isOverdue ? ' class="sctk-cron-overdue"' : ''; ?>>
								<td><?php echo esc_html($event['hook']); ?></td>
								<td title="<?php echo esc_attr((string) $datetimeFormatted); ?>">
									<?php echo esc_html($timeLabel); ?>
									<?php if ($isOverdue) : ?>
										<span class="sctk-cron-badge-overdue"><?php echo esc_html__('Overdue', Constants::TEXT_DOMAIN); ?></span>
									<?php endif; ?>
								</td>
								<td><?php echo esc_html($this->getScheduleLabel($event['schedule'])); ?></td>
								<td class="sctk-cron-args">
									<?php if (empty($event['args'])) : ?>
										<?php echo esc_html__('None', Constants::TEXT_DOMAIN); ?>
									<?php else : ?>
										<code><?php echo esc_html((string) wp_json_encode($event['args'])); ?></code>
									<?php endif; ?>
								</td>
								<td>
									<button
										type="button"
										class="sctk-cron-run"
										data-hook="<?php echo esc_attr($event['hook']); ?>"
										data-sig="<?php echo esc_attr($sig); ?>"
										data-timestamp="<?php echo esc_attr((string) $event['timestamp']); ?>"
										data-nonce="<?php echo esc_attr($nonce); ?>"
									>
										<?php echo esc_html__('Run Now', Constants::TEXT_DOMAIN); ?>
									</button>
								</td>
							</tr>
						<?php endforeach; ?>
					<?php endif; ?>
				</tbody>
			</table>
		</div>

		<script>
		document.querySelectorAll('.sctk-cron-run').forEach(function(btn) {
			btn.addEventListener('click', function() {
				var button = this;
				var originalText = button.textContent;

				button.disabled = true;
				button.innerHTML = '<?php echo esc_js(__('Running…', Constants::TEXT_DOMAIN)); ?>';

				var data = new FormData();
				data.append('action', 'sctk_run_cron');
				data.append('hook', button.dataset.hook);
				data.append('sig', button.dataset.sig);
				data.append('timestamp', button.dataset.timestamp);
				data.append('_wpnonce', button.dataset.nonce);

				fetch(ajaxurl, {
					method: 'POST',
					body: data,
					credentials: 'same-origin'
				})
				.then(function(response) { return response.json(); })
				.then(function(result) {
					if (result.success) {
						button.innerHTML = '<span class="sctk-cron-done">&#10003; <?php echo esc_js(__('Done', Constants::TEXT_DOMAIN)); ?></span>';
					} else {
						button.innerHTML = '<span class="sctk-cron-error"><?php echo esc_js(__('Error', Constants::TEXT_DOMAIN)); ?></span>';
					}

					setTimeout(function() {
						button.textContent = originalText;
						button.disabled = false;
					}, 2000);
				})
				.catch(function() {
					button.innerHTML = '<span class="sctk-cron-error"><?php echo esc_js(__('Error', Constants::TEXT_DOMAIN)); ?></span>';
					setTimeout(function() {
						button.textContent = originalText;
						button.disabled = false;
					}, 2000);
				});
			});
		});
		</script>
		<?php
	}

	/**
	 * Handle AJAX cron event execution.
	 *
	 * Validates the request, verifies permissions and nonce,
	 * executes the cron hook, and returns a JSON response.
	 */
	public function handleRunNowAjax(): void
	{
		if (!current_user_can('manage_options')) {
			wp_send_json_error(['message' => __('Unauthorized.', Constants::TEXT_DOMAIN)], 403);
		}

		$hook = isset($_POST['hook']) ? sanitize_text_field(wp_unslash($_POST['hook'])) : '';
		$sig = isset($_POST['sig']) ? sanitize_text_field(wp_unslash($_POST['sig'])) : '';
		$timestamp = isset($_POST['timestamp']) ? (int) $_POST['timestamp'] : 0;

		if ('' === $hook || '' === $sig || 0 === $timestamp) {
			wp_send_json_error(['message' => __('Missing parameters.', Constants::TEXT_DOMAIN)], 400);
		}

		check_ajax_referer('sctk_run_cron_' . $hook . '_' . $sig);

		$cronArray = _get_cron_array();
		if (!isset($cronArray[$timestamp][$hook])) {
			wp_send_json_error(['message' => __('Cron event not found.', Constants::TEXT_DOMAIN)], 404);
		}

		// Find the matching event by signature
		$matchedArgs = null;
		$matchedSchedule = '';

		foreach ($cronArray[$timestamp][$hook] as $eventSig => $eventData) {
			if ($sig === $eventSig) {
				$matchedArgs = $eventData['args'];
				$matchedSchedule = $eventData['schedule'] ?? '';
				break;
			}
		}

		if (null === $matchedArgs) {
			wp_send_json_error(['message' => __('Cron event not found.', Constants::TEXT_DOMAIN)], 404);
		}

		// Execute the cron hook
		do_action_ref_array($hook, $matchedArgs);

		// Unschedule one-time events (recurring events reschedule automatically)
		if ('' === $matchedSchedule || false === $matchedSchedule) {
			wp_unschedule_event($timestamp, $hook, $matchedArgs);
		}

		wp_send_json_success();
	}

	/**
	 * Get all cron events as a flat sorted array.
	 *
	 * Retrieves the WordPress cron array and flattens it into
	 * a list of individual events sorted by timestamp ascending.
	 *
	 * @return array<int, array{timestamp: int, hook: string, args: array<mixed>, schedule: string, interval: int}> Sorted cron events.
	 */
	private function getCronEvents(): array
	{
		$cronArray = _get_cron_array();
		if ([] === $cronArray) {
			return [];
		}

		$events = [];

		foreach ($cronArray as $timestamp => $hooks) {
			foreach ($hooks as $hook => $entries) {
				foreach ($entries as $data) {
					$events[] = [
						'timestamp' => (int) $timestamp,
						'hook' => $hook,
						'args' => $data['args'] ?? [],
						'schedule' => $data['schedule'] ?? '',
						'interval' => $data['interval'] ?? 0,
					];
				}
			}
		}

		usort($events, static fn(array $a, array $b): int => $a['timestamp'] <=> $b['timestamp']);

		return $events;
	}

	/**
	 * Get human-readable schedule label.
	 *
	 * Translates a cron schedule key into its display name
	 * as registered in WordPress.
	 *
	 * @param string $schedule The cron schedule key.
	 * @return string Human-readable schedule label.
	 */
	private function getScheduleLabel(string $schedule): string
	{
		if ('' === $schedule) {
			return __('Once', Constants::TEXT_DOMAIN);
		}

		$schedules = wp_get_schedules();

		return $schedules[$schedule]['display'] ?? __('Once', Constants::TEXT_DOMAIN);
	}

	/**
	 * Check if a cron event is overdue.
	 *
	 * @param int $timestamp The scheduled timestamp.
	 * @return bool True if the event is past its scheduled time.
	 */
	private function isOverdue(int $timestamp): bool
	{
		return $timestamp < time();
	}
}
