<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features\DevTools;

use SignocoreToolkit\Application\Constants;
use SignocoreToolkit\Infrastructure\Traits\Singleton;

/**
 * Developer tools admin menu integration.
 *
 * @package SignocoreToolkit\Features\DevTools
 * @since 3.0.0
 */
final class AdminMenu
{
	use Singleton;

	/**
	 * Required capability for menu access.
	 */
	private string $capability = 'manage_options';

	/**
	 * Initialize admin menu features.
	 *
	 * Registers the submenu pages and settings hooks.
	 */
	protected function init(): void
	{
		add_action('admin_menu', [$this, 'registerSubmenus']);
		add_action('admin_init', [$this, 'registerSettings']);
	}

	/**
	 * Register developer tool submenu pages.
	 *
	 * These pages are hidden from the sidebar menu (null parent) and
	 * accessed via tab navigation on the DevTools settings page.
	 */
	public function registerSubmenus(): void
	{
		// Hidden pages - accessible via DevTools tabs only
		// Using 'options.php' as a non-existent parent hides from menu while registering the page
		$hiddenParent = 'options.php';

		add_submenu_page(
			$hiddenParent,
			esc_html__('Mail Log', Constants::TEXT_DOMAIN),
			esc_html__('Mail Log', Constants::TEXT_DOMAIN),
			$this->capability,
			'signocore-toolkit-mail-log',
			[MailCatcher::load(), 'renderPage']
		);

		add_submenu_page(
			$hiddenParent,
			esc_html__('Transients', Constants::TEXT_DOMAIN),
			esc_html__('Transients', Constants::TEXT_DOMAIN),
			$this->capability,
			'signocore-toolkit-transients',
			[TransientViewer::load(), 'renderPage']
		);

		add_submenu_page(
			$hiddenParent,
			esc_html__('Cron Jobs', Constants::TEXT_DOMAIN),
			esc_html__('Cron Jobs', Constants::TEXT_DOMAIN),
			$this->capability,
			'signocore-toolkit-cron',
			[CronViewer::load(), 'renderPage']
		);

		add_submenu_page(
			$hiddenParent,
			esc_html__('System Status', Constants::TEXT_DOMAIN),
			esc_html__('System Status', Constants::TEXT_DOMAIN),
			$this->capability,
			'signocore-toolkit-status',
			[SystemStatus::load(), 'renderPage']
		);
	}

	/**
	 * Register toolkit settings using the WordPress Settings API.
	 *
	 * Registers settings for environment override, mail log mode,
	 * prevent sending, and auto-delete days. These settings use
	 * the sctk_ prefix for backward compatibility.
	 */
	public function registerSettings(): void
	{
		register_setting('signocore-toolkit-devtools', 'sctk_environment_override', [
			'type' => 'string',
			'default' => 'auto',
			'sanitize_callback' => [$this, 'sanitizeEnvironmentOverride'],
		]);

		register_setting('signocore-toolkit-devtools', 'sctk_mail_log_mode', [
			'type' => 'string',
			'default' => 'all',
			'sanitize_callback' => [$this, 'sanitizeLogMode'],
		]);

		register_setting('signocore-toolkit-devtools', 'sctk_mail_prevent_sending', [
			'type' => 'boolean',
			'default' => false,
			'sanitize_callback' => 'rest_sanitize_boolean',
		]);

		register_setting('signocore-toolkit-devtools', 'sctk_mail_auto_delete_days', [
			'type' => 'integer',
			'default' => 30,
			'sanitize_callback' => 'absint',
		]);
	}

	/**
	 * Sanitize the environment override value.
	 *
	 * @param mixed $value The submitted value.
	 * @return string Sanitized environment value.
	 */
	public function sanitizeEnvironmentOverride(mixed $value): string
	{
		$allowed = ['auto', 'development', 'staging', 'production'];

		if (in_array($value, $allowed, true)) {
			return $value;
		}

		return 'auto';
	}

	/**
	 * Sanitize the log mode value.
	 *
	 * @param mixed $value The submitted value.
	 * @return string Sanitized log mode value.
	 */
	public function sanitizeLogMode(mixed $value): string
	{
		$allowed = ['all', 'blocked'];

		if (in_array($value, $allowed, true)) {
			return $value;
		}

		return 'all';
	}
}
