<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features\DevTools;

use SignocoreToolkit\Application\Constants;
use SignocoreToolkit\Infrastructure\Traits\Singleton;

/**
 * Admin bar performance stats display.
 *
 * @package SignocoreToolkit\Features\DevTools
 * @since 3.0.0
 */
final class AdminBarStats
{
	use Singleton;

	/**
     * Initialize admin bar stats feature.
     *
     * Registers hooks to add performance stats to the admin bar
     * and inject required styles.
     */
    protected function init(): void
	{
		add_action('admin_bar_menu', [$this, 'addStatsToAdminBar'], 999);
		add_action('admin_head', [$this, 'renderStyles']);
		add_action('wp_head', [$this, 'renderStyles']);
	}

	/**
	 * Add performance stats node and detail dropdown to the admin bar.
	 *
	 * Displays a compact summary in the bar and detailed metrics
	 * with labels in a dropdown on hover.
	 *
	 * @param \WP_Admin_Bar $adminBar WordPress admin bar instance.
	 */
	public function addStatsToAdminBar(\WP_Admin_Bar $adminBar): void
	{
		if (!current_user_can('manage_options')) {
			return;
		}

		$queries = $this->getQueryCount();
		$memory = $this->getPeakMemory();
		$time = $this->getLoadTime();

		$adminBar->add_node([
			'id' => 'sctk-stats',
			'title' => sprintf('%dQ | %s | %ss', $queries, $memory, $time),
			'meta' => ['class' => 'sctk-admin-bar-stats'],
		]);

		$details = [
			'queries' => [
				'label' => __('DB Queries', Constants::TEXT_DOMAIN),
				'value' => (string) $queries,
			],
			'memory' => [
				'label' => __('Peak Memory', Constants::TEXT_DOMAIN),
				'value' => sprintf('%s / %s', $memory, $this->getMemoryLimit()),
			],
			'time' => [
				'label' => __('Load Time', Constants::TEXT_DOMAIN),
				'value' => $time . 's',
			],
			'php' => [
				'label' => __('PHP Version', Constants::TEXT_DOMAIN),
				'value' => PHP_VERSION,
			],
			'wp' => [
				'label' => __('WordPress', Constants::TEXT_DOMAIN),
				'value' => get_bloginfo('version'),
			],
		];

		if (defined('WC_VERSION')) {
			$details['woo'] = [
				'label' => __('WooCommerce', Constants::TEXT_DOMAIN),
				'value' => WC_VERSION,
			];
		}

		foreach ($details as $id => $detail) {
			$adminBar->add_node([
				'id' => 'sctk-stats-' . $id,
				'parent' => 'sctk-stats',
				'title' => sprintf(
					'<span class="sctk-stats-label">%s</span><span class="sctk-stats-value">%s</span>',
					esc_html($detail['label']),
					esc_html($detail['value'])
				),
				'meta' => ['class' => 'sctk-stats-detail'],
			]);
		}
	}

	/**
     * Render inline styles for the admin bar stats node.
     *
     * Outputs a style block that formats the stats display
     * with monospace font and reduced opacity.
     */
    public function renderStyles(): void
	{
		if (!current_user_can('manage_options') || !is_admin_bar_showing()) {
			return;
		}

		?>
		<style>
			#wpadminbar .sctk-admin-bar-stats > .ab-item {
				font-size: 12px !important;
				opacity: 0.6;
				letter-spacing: 0.02em;
				line-height: 32px !important;
				height: 32px !important;
			}
			#wpadminbar .sctk-admin-bar-stats:hover > .ab-item {
				opacity: 1;
			}
			#wpadminbar .sctk-stats-detail .ab-item {
				display: flex !important;
				justify-content: space-between !important;
				gap: 16px;
				padding: 0 12px !important;
				line-height: 32px !important;
				height: 32px !important;
				cursor: default !important;
			}
			#wpadminbar .sctk-stats-detail .ab-item:hover {
				background: transparent !important;
				color: inherit !important;
			}
			#wpadminbar .sctk-stats-label {
				color: #999;
				font-size: 11px;
			}
			#wpadminbar .sctk-stats-value {
				color: #fff;
				font-size: 12px;
				font-weight: 600;
			}
			@media screen and (max-width: 782px) {
				#wpadminbar .sctk-admin-bar-stats > .ab-item {
					line-height: 46px !important;
					height: 46px !important;
				}
				#wpadminbar .sctk-stats-detail .ab-item {
					line-height: 46px !important;
					height: 46px !important;
				}
			}
		</style>
		<?php
	}

	/**
     * Get the total number of database queries executed.
     *
     * @return int Number of queries.
     */
    private function getQueryCount(): int
	{
		global $wpdb;

		return $wpdb->num_queries;
	}

	/**
     * Get the peak memory usage formatted as a human-readable string.
     *
     * @return string Formatted memory usage (e.g., "12 MB").
     */
    private function getPeakMemory(): string
	{
		return size_format(memory_get_peak_usage(true));
	}

	/**
	 * Get the page load time in seconds.
	 *
	 * Calculates elapsed time from the request start to the current moment.
	 *
	 * @return string Load time formatted to 3 decimal places.
	 */
	private function getLoadTime(): string
	{
		$time = microtime(true) - $_SERVER['REQUEST_TIME_FLOAT'];

		return number_format($time, 3);
	}

	/**
	 * Get the PHP memory limit as a human-readable string.
	 *
	 * @return string Formatted memory limit (e.g., "256 MB").
	 */
	private function getMemoryLimit(): string
	{
		$limit = ini_get('memory_limit');

		if ($limit === '' || $limit === '-1') {
			return (string) __('Unlimited', Constants::TEXT_DOMAIN);
		}

		$formatted = size_format(wp_convert_hr_to_bytes($limit));

		return $formatted !== false ? $formatted : $limit;
	}
}
