<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features;

use SignocoreToolkit\Infrastructure\Traits\Singleton;
use SignocoreToolkit\Application\Constants;
use SignocoreToolkit\Application\OptionsManager;

/**
 * GDPR-compliant cookie consent management.
 *
 * Provides granular consent for cookie categories (necessary, analytics, marketing)
 * with default deny for non-essential cookies and the ability to change preferences.
 */
final class Cookies
{
	use Singleton;

	private const COOKIE_NAME = 'sctk_consent';

	protected function init(): void
	{
		// Inject default consent script in head (before any tracking scripts)
		add_action('wp_head', [$this, 'injectDefaultConsent'], 1);

		// Render consent UI in footer
		add_action('wp_footer', [$this, 'renderConsentUI'], 99);
	}

	/**
	 * Get consent categories with labels and descriptions.
	 *
	 * @return array<string, array{label: string, description: string, required: bool}>
	 */
	public function getConsentCategories(): array
	{
		$categories = [
			'necessary' => [
				'label' => __('Necessary', Constants::TEXT_DOMAIN),
				'description' => __('Required for the website to function properly. These cookies cannot be disabled.', Constants::TEXT_DOMAIN),
				'required' => true,
			],
			'analytics' => [
				'label' => __('Analytics', Constants::TEXT_DOMAIN),
				'description' => __('Help us understand how visitors interact with our website to improve user experience.', Constants::TEXT_DOMAIN),
				'required' => false,
			],
			'marketing' => [
				'label' => __('Marketing', Constants::TEXT_DOMAIN),
				'description' => __('Used for advertising and personalized content based on your interests.', Constants::TEXT_DOMAIN),
				'required' => false,
			],
		];

		/** @var array<string, array{label: string, description: string, required: bool}> */
		return apply_filters('signocore_toolkit_consent_categories', $categories);
	}

	/**
	 * Inject default consent state script.
	 *
	 * Must run before any tracking scripts to ensure default deny.
	 */
	public function injectDefaultConsent(): void
	{
		if (Constants::$isAdmin || !$this->isEnabled()) {
			return;
		}

		$script = "window.dataLayer = window.dataLayer || [];
function gtag(){dataLayer.push(arguments);}
gtag('consent', 'default', {
	'analytics_storage': 'denied',
	'ad_storage': 'denied',
	'ad_user_data': 'denied',
	'ad_personalization': 'denied',
	'functionality_storage': 'granted',
	'security_storage': 'granted'
});";

		echo '<script>' . $script . '</script>' . "\n";
	}

	/**
	 * Render the consent UI (banner, modal, floating button).
	 */
	public function renderConsentUI(): void
	{
		if (Constants::$isAdmin || !$this->isEnabled()) {
			return;
		}

		// Skip if previewing forms (Ninja Forms, etc.)
		$nfPreview = isset($_GET['nf_preview_form']) ? sanitize_text_field(wp_unslash($_GET['nf_preview_form'])) : '';
		if (!empty($nfPreview)) {
			return;
		}

		$this->renderBanner();
		$this->renderPreferencesModal();
		$this->renderFloatingButton();
		$this->renderConsentData();
	}

	/**
	 * Render the consent banner HTML.
	 */
	private function renderBanner(): void
	{
		$position = OptionsManager::getString('cookie_banner_position', 'bottom-right');
		$customText = OptionsManager::getString('cookie_banner_text', '');

		$defaultText = __('We use cookies for analytics and marketing purposes. You can customize your preferences or accept all cookies.', Constants::TEXT_DOMAIN);
		$bannerText = !empty($customText) ? $customText : $defaultText;

		$privacyLink = $this->getPrivacyPolicyLink();

		$html = '<div id="sctk-cookie-banner" class="sctk-cookie-banner sctk-cookie-banner--' . esc_attr($position) . '" role="dialog" aria-modal="true" aria-labelledby="sctk-cookie-title" style="display:none;">
	<div class="sctk-cookie-banner__content">
		<div class="sctk-cookie-banner__header">
			<span class="sctk-cookie-banner__icon" aria-hidden="true">' . $this->getCookieIcon() . '</span>
			<h2 id="sctk-cookie-title" class="sctk-cookie-banner__title">' . esc_html__('Cookie Consent', Constants::TEXT_DOMAIN) . '</h2>
		</div>
		<p class="sctk-cookie-banner__text">' . esc_html($bannerText) . $privacyLink . '</p>
		<div class="sctk-cookie-banner__actions">
			<button type="button" id="sctk-accept-all" class="sctk-cookie-banner__btn sctk-cookie-banner__btn--primary">' . esc_html__('Accept All', Constants::TEXT_DOMAIN) . '</button>
			<button type="button" id="sctk-reject-all" class="sctk-cookie-banner__btn sctk-cookie-banner__btn--secondary">' . esc_html__('Reject All', Constants::TEXT_DOMAIN) . '</button>
			<button type="button" id="sctk-customize" class="sctk-cookie-banner__btn sctk-cookie-banner__btn--link" style="background:transparent;border:none;box-shadow:none;padding:0.5rem;color:#546e7a;">' . esc_html__('Customize', Constants::TEXT_DOMAIN) . '</button>
		</div>
	</div>
</div>';

		/** @var string */
		$filteredHtml = apply_filters('signocore_toolkit_consent_banner_html', $html);
		echo $filteredHtml;
	}

	/**
	 * Render the preferences modal HTML.
	 */
	private function renderPreferencesModal(): void
	{
		$categories = $this->getConsentCategories();

		$categoriesHtml = '';
		foreach ($categories as $key => $category) {
			$isRequired = $category['required'];
			$toggleHtml = $isRequired
				? '<span class="sctk-cookie-category__always-on">' . esc_html__('Always active', Constants::TEXT_DOMAIN) . '</span>'
				: '<label class="sctk-cookie-category__toggle">
					<input type="checkbox" name="sctk_consent_' . esc_attr($key) . '" id="sctk-consent-' . esc_attr($key) . '" value="1">
					<span class="sctk-cookie-category__slider"></span>
				</label>';

			$categoriesHtml .= '<div class="sctk-cookie-category' . ($isRequired ? ' sctk-cookie-category--required' : '') . '">
	<div class="sctk-cookie-category__header">
		<span class="sctk-cookie-category__label">' . esc_html($category['label']) . '</span>
		' . $toggleHtml . '
	</div>
	<p class="sctk-cookie-category__description">' . esc_html($category['description']) . '</p>
</div>';
		}

		echo '<div id="sctk-cookie-modal" class="sctk-cookie-modal" role="dialog" aria-modal="true" aria-labelledby="sctk-modal-title" style="display:none;">
	<div class="sctk-cookie-modal__backdrop" data-action="close"></div>
	<div class="sctk-cookie-modal__content">
		<div class="sctk-cookie-modal__header">
			<h2 id="sctk-modal-title" class="sctk-cookie-modal__title">' . esc_html__('Cookie Preferences', Constants::TEXT_DOMAIN) . '</h2>
			<button type="button" class="sctk-cookie-modal__close" data-action="close" aria-label="' . esc_attr__('Close', Constants::TEXT_DOMAIN) . '" style="all:unset;display:flex;align-items:center;justify-content:center;width:2rem;height:2rem;background:transparent;border:none;box-shadow:none;cursor:pointer;color:#94a3b8;">&times;</button>
		</div>
		<div class="sctk-cookie-modal__body">
			<p class="sctk-cookie-modal__intro">' . esc_html__('Manage your cookie preferences below. Necessary cookies are always active as they are essential for the website to function.', Constants::TEXT_DOMAIN) . '</p>
			' . $categoriesHtml . '
		</div>
		<div class="sctk-cookie-modal__footer">
			<button type="button" id="sctk-save-preferences" class="sctk-cookie-banner__btn sctk-cookie-banner__btn--secondary">' . esc_html__('Save Preferences', Constants::TEXT_DOMAIN) . '</button>
			<button type="button" id="sctk-modal-accept-all" class="sctk-cookie-banner__btn sctk-cookie-banner__btn--primary">' . esc_html__('Accept All', Constants::TEXT_DOMAIN) . '</button>
		</div>
	</div>
</div>';
	}

	/**
	 * Render the floating preferences button.
	 */
	private function renderFloatingButton(): void
	{
		$showButton = OptionsManager::getBool('cookie_show_floating_button', true);
		if (!$showButton) {
			return;
		}

		$position = OptionsManager::getString('cookie_banner_position', 'bottom-right');

		echo '<button type="button" id="sctk-cookie-floating" class="sctk-cookie-floating sctk-cookie-floating--' . esc_attr($position) . '" aria-label="' . esc_attr__('Cookie Preferences', Constants::TEXT_DOMAIN) . '" style="display:none;">
	' . $this->getCookieIcon() . '
</button>';
	}

	/**
	 * Output consent configuration data for JavaScript.
	 */
	private function renderConsentData(): void
	{
		$expiryDays = OptionsManager::getInt('cookie_expiry_days', 365);
		$showFloating = OptionsManager::getBool('cookie_show_floating_button', true);

		$config = [
			'cookieName' => self::COOKIE_NAME,
			'expiryDays' => $expiryDays,
			'showFloating' => $showFloating,
			'version' => 1,
		];

		echo '<script>window.sctkConsentConfig = ' . wp_json_encode($config) . ';</script>' . "\n";
	}

	/**
	 * Get privacy policy link HTML if configured.
	 */
	private function getPrivacyPolicyLink(): string
	{
		$privacyPolicyId = get_option('wp_page_for_privacy_policy');
		if (empty($privacyPolicyId)) {
			return '';
		}

		$permalink = get_permalink($privacyPolicyId);
		if (!$permalink) {
			return '';
		}

		return ' <a href="' . esc_url((string) $permalink) . '" class="sctk-cookie-banner__link">' . esc_html__('Privacy Policy', Constants::TEXT_DOMAIN) . '</a>';
	}

	/**
	 * Check if cookie consent is enabled.
	 */
	private function isEnabled(): bool
	{
		return OptionsManager::getBool('add_cookie_notice', true);
	}

	/**
	 * Get cookie SVG icon.
	 */
	private function getCookieIcon(): string
	{
		return '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><circle cx="8" cy="9" r="1" fill="currentColor"/><circle cx="15" cy="8" r="1" fill="currentColor"/><circle cx="10" cy="14" r="1" fill="currentColor"/><circle cx="15" cy="14" r="1" fill="currentColor"/><circle cx="6" cy="12" r="1" fill="currentColor"/></svg>';
	}
}
