<?php

declare(strict_types=1);

namespace SignocoreToolkit\Features;

use SignocoreToolkit\Support\FAQHelper;
use SignocoreToolkit\Support\GlossaryHelper;
use SignocoreToolkit\Support\TaxonomyHelper;
use Kadence_Blocks_Posts_Block;
use SignocoreToolkit\Infrastructure\Traits\Singleton;
use SignocoreToolkit\Application\Constants;

final class Blog
{
    use Singleton;

    protected function init(): void
    {
        add_action('save_post', [$this, 'setFallbackFeaturedImage'], 10, 2);
        add_action('kadence_single_after_inner_content', [$this, 'addRelatedTerms']);
        add_action('kadence_single_after_inner_content', [$this, 'addRelatedQuestions']);
        add_action('kadence_after_main_content', [$this, 'addRelatedPostsToTerms'], 20);
        add_action('kadence_after_main_content', [$this, 'addRelatedPostsToQuestions'], 20);
        add_filter('get_the_archive_title', [$this, 'replaceArchiveTitle'], 11);

        add_action('wp_before_load_template', function ($_template_file): void {
            if (TaxonomyHelper::shouldReplaceContent($_template_file)) {
                ob_start();
            }
        });

        add_action('wp_after_load_template', function ($_template_file): void {
            if (TaxonomyHelper::shouldReplaceContent($_template_file)) {
                ob_end_clean();
                include_once Constants::TEMPLATE_DIR . 'archive-no-posts.php';
            }
        });

        if (Constants::$isAdmin) {
            add_filter('replace_editor', [$this, 'addBlogPageEditor'], 10, 2);
            add_action('show_user_profile', [$this, 'renderAuthorIntro']);
            add_action('edit_user_profile', [$this, 'renderAuthorIntro']);
            add_action('profile_update', [$this, 'saveAuthorIntro']);

            add_action('init', function (): void {
                $allTaxonomies = TaxonomyHelper::getTaxonomies();
                $allTaxonomies = $allTaxonomies === [] ? ['category', 'post_tag'] : array_keys($allTaxonomies);

                foreach ($allTaxonomies as $taxonomy) {
                    add_action($taxonomy . '_edit_form_fields', [$this, 'renderTermIntro']);
                    add_action('edited_' . $taxonomy, [$this, 'saveTermIntro']);
                }
            });
        }
    }

    public function addBlogPageEditor(bool $replace, \WP_Post $post): bool
    {
        $blogId = (int) get_option('page_for_posts');
        if (!$replace && $post->ID === $blogId) {
            $post->post_content = '<!--non-empty-content-->';
        }

        return $replace;
    }

    public function setFallbackFeaturedImage(int $postId, \WP_Post $post): void
    {
        if ($post->post_type !== 'post' || (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE)) {
            return;
        }

        $defaultFeaturedImage = get_option(Constants::OPTION_PREFIX . 'default_featured_image', '');
        if (empty($defaultFeaturedImage)) {
            return;
        }

        if (!has_post_thumbnail($postId)) {
            set_post_thumbnail($postId, $defaultFeaturedImage);
        }
    }

    public function addRelatedQuestions(): void
    {
        if (!defined('SIGNOCORE_FAQ_VERSION') || !is_singular('post')) {
            return;
        }

        $showRelatedQuestions = (bool) get_option(Constants::OPTION_PREFIX . 'show_related_questions', true);
        if (!$showRelatedQuestions) {
            return;
        }

        $post = get_post();
        $relatedQuestions = get_transient('related_questions_' . $post->ID);

        if ($relatedQuestions === false) {
            $tags = wp_get_post_tags($post->ID, ['taxonomy' => 'post_tag', 'fields' => 'slugs']);

            if (empty($tags) || is_wp_error($tags)) {
                return;
            }

            /** @var array<int, string> $tags */
            $faqs = FAQHelper::faqsByTags($tags, 10);
            $relatedQuestions = '';

            if ($faqs !== []) {
                $relatedQuestions =  FAQHelper::render($faqs, sprintf(__('Questions related to %s', Constants::TEXT_DOMAIN), esc_html($post->post_title)), 'related-questions');

                set_transient('related_questions_' . $post->ID, $relatedQuestions, MONTH_IN_SECONDS);
            }
        }

        echo wp_kses_post($relatedQuestions);
    }

    public function addRelatedPostsToTerms(): void
    {
        if (!defined('SIGNOCORE_GLOSSARY_VERSION') || !is_singular('wpsseo_term') || !class_exists('Kadence_Blocks_Posts_Block')) {
            return;
        }

        $postId = get_the_ID();
        if ($postId === false) {
            return;
        }

        $tags = wp_get_post_terms($postId, 'post_tag', ['fields' => 'ids']);
        if (empty($tags) || is_wp_error($tags)) {
            return;
        }

        /** @var list<int> $tags */
        $attributes = [
            'postType'       => 'post',
            'postsToShow'    => 2,
            'columns'        => 2,
            'orderBy'        => 'rand',
            'allowSticky'    => false,
            'meta'           => false,
            'excerpt'        => false,
            'readmore'       => false,
            'titleFont' => [
                [
                    'level' => 3
                ]
            ],
            'loopStyle'      => 'boxed',
            'uniqueID'       => 'signocore-related-posts',
            'tags'           => array_map(fn(int $id): array => ['value' => $id], $tags),
            'excludeTax'     => '',
            'customKadenceArchiveColors' => [],
            'customKadenceTaxonomyBackgrounds' => [],
            'taxonomyType' => '',
        ];

        $relatedPosts = Kadence_Blocks_Posts_Block::get_instance()->build_html(
            $attributes,
            'relatedPosts',
            '',
            null
        );

        if (!empty($relatedPosts) && is_string($relatedPosts) && stripos(strip_tags($relatedPosts), (string) __('No posts', 'kadence-blocks')) === false) {
            echo '<section class="signocore-related-posts">';
            echo '<h2>' . esc_html__('Related Articles', Constants::TEXT_DOMAIN) . '</h2>';
            echo wp_kses_post($relatedPosts);
            echo '</section>';
        }
    }

    public function addRelatedPostsToQuestions(): void
    {
        if (!defined('SIGNOCORE_FAQ_VERSION') || !is_singular('wpsseo_faq') || !class_exists('Kadence_Blocks_Posts_Block')) {
            return;
        }

        $postId = get_the_ID();
        if ($postId === false) {
            return;
        }

        $tags = wp_get_post_terms($postId, 'post_tag', ['fields' => 'ids']);
        if (empty($tags) || is_wp_error($tags)) {
            return;
        }

        /** @var list<int> $tags */
        $attributes = [
            'postType'       => 'post',
            'postsToShow'    => 2,
            'columns'        => 2,
            'orderBy'        => 'rand',
            'allowSticky'    => false,
            'meta'           => false,
            'excerpt'        => false,
            'readmore'       => false,
            'titleFont' => [
                [
                    'level' => 3
                ]
            ],
            'loopStyle'      => 'boxed',
            'uniqueID'       => 'signocore-related-posts',
            'tags'           => array_map(fn(int $id): array => ['value' => $id], $tags),
            'excludeTax'     => '',
            // Tilføj disse for at undgå warnings:
            'customKadenceArchiveColors' => [],
            'customKadenceTaxonomyBackgrounds' => [],
            'taxonomyType' => '',
        ];

        $relatedPosts = Kadence_Blocks_Posts_Block::get_instance()->build_html(
            $attributes,
            'relatedPosts',
            '',
            null
        );

        if (!empty($relatedPosts) && is_string($relatedPosts) && stripos(strip_tags($relatedPosts), (string) __('No posts', 'kadence-blocks')) === false) {
            echo '<section class="signocore-related-posts">';
            echo '<h2>' . esc_html__('Related Articles', Constants::TEXT_DOMAIN) . '</h2>';
            echo wp_kses_post($relatedPosts);
            echo '</section>';
        }
    }

    public function addRelatedTerms(): void
    {
        if (!defined('SIGNOCORE_GLOSSARY_VERSION') || !is_singular('post')) {
            return;
        }

        $showRelatedGlossaryTerms = (bool) get_option(Constants::OPTION_PREFIX . 'show_related_glossary_terms', true);
        if (!$showRelatedGlossaryTerms) {
            return;
        }

        $post = get_post();
        $relatedTerms = get_transient('related_terms_' . $post->ID);
        $relatedTerms = false;

        $tags = get_the_terms($post->ID, 'post_tag');
        if (empty($tags) || is_wp_error($tags)) {
            return;
        }

        $tagSlugs = [];
        $tagNames = [];
        foreach ($tags as $tag) {
            $tagSlugs[] = $tag->slug;
            $tagNames[] = $tag->name;
        }

        $termsByTitle = [];
        $excludeIds = [];
        $batchResults = GlossaryHelper::termsByTitles($tagNames);
        foreach ($batchResults as $terms) {
            foreach ($terms as $term) {
                if (!isset($excludeIds[$term->ID])) {
                    $termsByTitle[] = $term;
                    $excludeIds[$term->ID] = true;
                }
            }
        }

        $termsByTags = GlossaryHelper::termsByTags($tagSlugs, 10, array_keys($excludeIds));
        $allTerms = array_merge($termsByTitle, $termsByTags);

        if ($allTerms !== []) {
            update_postmeta_cache(array_column($allTerms, 'ID'));
        }

        usort($allTerms, function ($a, $b): int {
            $aViews = get_post_meta($a->ID, 'page_views', true) ?: 0;
            $bViews = get_post_meta($b->ID, 'page_views', true) ?: 0;
            return $bViews <=> $aViews;
        });
        $allTerms = array_slice($allTerms, 0, 10);
        $relatedTerms = '';
        if ($allTerms !== []) {
            $relatedTerms = GlossaryHelper::render($allTerms, sprintf(__('Terms related to %s', Constants::TEXT_DOMAIN), esc_html($post->post_title)), 'related-words');

            set_transient('related_terms_' . $post->ID, $relatedTerms, MONTH_IN_SECONDS);
        }

        echo wp_kses_post($relatedTerms);
    }

    public function replaceArchiveTitle(string $title): string
    {
        $objectId = get_queried_object_id();

        if (empty($objectId)) {
            return $title;
        }

        $headline = is_author() ? get_user_meta($objectId, 'headline', true) : get_term_meta($objectId, 'headline', true);
        if ($headline) {
            return $headline;
        }

        return $title;
    }

    public function renderAuthorIntro(object $user): void
    {
        if (empty($user->ID)) {
            return;
        }

        wp_nonce_field('save_author_intro', 'save_author_intro');

        $introTitle = get_user_meta($user->ID, 'headline', true);
        $introDescription = get_user_meta($user->ID, 'intro_text', true);
        $ppDisabled = (bool) get_user_meta($user->ID, 'pp_disabled', true) ? 'checked="checked"' : '';

        echo '<div class="postbox signocore-theme"><div class="inside"><h2>' . __('Author Intro', Constants::TEXT_DOMAIN) . '</h2>
        <table class="form-table">
            <tbody>
                <tr>
                    <th scope="row"><label for="signocore_title">' . __('Author Headline', Constants::TEXT_DOMAIN) . '</label></th>
                    <td>
                        <p><input class="regular-text" type="text" name="signocore_intro[headline]" id="signocore_headline" value="' . esc_attr($introTitle) . '"></p>
                        <span class="description">' . sprintf(esc_html__('Will display in the %s tag at the top of the first page', 'genesis'), '<code>&lt;h1&gt;</code>') . '</span>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="signocore_intro_text">' . __('Author Description', Constants::TEXT_DOMAIN) . '</label></th>
                    <td style="max-width: 720px; display: block;">';
        wp_editor($introDescription, 'signocore_intro_text', ['media_buttons' => false, 'textarea_name' => 'signocore_intro[intro_text]', 'textarea_rows' => 15]);
        echo '<span class="description">' . esc_html__('This text will be the first paragraph, and display on the first page', 'genesis') . '</span>
                    </td>
                </tr>

                <tr>
                    <th scope="row">' . __('Author Settings', Constants::TEXT_DOMAIN) . '</th>
                    <td>
                        <p>
                            <label for="signocore_pp_disabled" class="signocore-theme-check-switch">
                                <input type="checkbox" name="signocore_intro[pp_disabled]" id="signocore_pp_disabled" value="1" ' . $ppDisabled . '>
								<span class="slider"></span>
                            </label>
                            ' . __('Hide profile picture for this author', Constants::TEXT_DOMAIN) . '
                        </p>
                    </td>
                </tr>

            </tbody>
        </table></div></div>';
    }

    public function saveAuthorIntro(int $userId): void
    {
        $nonce = isset($_POST['save_author_intro'])
            ? sanitize_text_field(wp_unslash($_POST['save_author_intro']))
            : '';

        if (!wp_verify_nonce($nonce, 'save_author_intro')) {
            return;
        }

        if (!current_user_can('edit_user', $userId)) {
            return;
        }

        $allowedKeys = ['signocore_title', 'signocore_intro_text', 'pp_disabled'];
        $introSettings = isset($_POST['signocore_intro']) && is_array($_POST['signocore_intro'])
            ? (array) wp_unslash($_POST['signocore_intro'])
            : [];

        if (!isset($introSettings['pp_disabled'])) {
            $introSettings['pp_disabled'] = 0;
        }

        foreach ($introSettings as $key => $value) {
            if (!in_array($key, $allowedKeys, true)) {
                continue;
            }
            $sanitized = ($key === 'signocore_intro_text') ? wp_kses_post($value) : sanitize_text_field($value);
            update_user_meta($userId, $key, $sanitized);
        }
    }

    public function renderTermIntro(object $term): void
    {
        if (empty($term->term_id)) {
            return;
        }

        wp_nonce_field('save_term_intro', 'save_term_intro');

        $introTitle = get_term_meta($term->term_id, 'headline', true);
        $introDescription = get_term_meta($term->term_id, 'intro_text', true);
        $additionalIntroText = get_term_meta($term->term_id, 'additional_intro_text', true);

        echo '</table><div class="postbox wpsseo"><div class="inside"><h2>' . __('Archive Intro', Constants::TEXT_DOMAIN) . '</h2>
        <table class="form-table">
            <tbody>

                <tr>
                    <th scope="row"><label for="signocore_headline">' . __('Archive Headline', Constants::TEXT_DOMAIN) . '</label></th>
                    <td>
                        <p><input class="large-text" type="text" name="signocore_intro[headline]" id="signocore_headline" value="' . esc_attr($introTitle) . '"></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="signocore_intro_text">' . __('Archive Intro Text', Constants::TEXT_DOMAIN) . '</label></th>
                    <td style="max-width: 720px; display: block;">';
        wp_editor($introDescription, 'signocore_intro_text', ['media_buttons' => false, 'textarea_name' => 'signocore_intro[intro_text]', 'textarea_rows' => 15]);
        echo '</td>
                </tr>

                <tr>
                    <th scope="row"><label for="signocore_additional_intro_text">' . __('Archive Additional Text (After Products/Posts)', Constants::TEXT_DOMAIN) . '</label></th>
                    <td style="max-width: 720px; display: block;">';
        wp_editor($additionalIntroText, 'signocore_additional_intro_text', ['media_buttons' => false, 'textarea_name' => 'signocore_intro[additional_intro_text]', 'textarea_rows' => 15]);
        echo '</td>
                </tr>

            </tbody>
        </table></div></div>';
    }

    public function saveTermIntro(int $termId): void
    {
        $nonce = isset($_POST['save_term_intro'])
            ? sanitize_text_field(wp_unslash($_POST['save_term_intro']))
            : '';

        if (!wp_verify_nonce($nonce, 'save_term_intro')) {
            return;
        }

        if (!current_user_can('edit_posts')) {
            return;
        }

        $allowedKeys = ['headline', 'intro_text', 'additional_intro_text'];
        $introSettings = isset($_POST['signocore_intro']) && is_array($_POST['signocore_intro'])
            ? (array) wp_unslash($_POST['signocore_intro'])
            : [];

        foreach ($introSettings as $key => $value) {
            if (!in_array($key, $allowedKeys, true)) {
                continue;
            }
            $sanitized = ($key === 'headline') ? sanitize_text_field($value) : wp_kses_post($value);
            update_term_meta($termId, $key, $sanitized);
        }
    }
}
