<?php

declare(strict_types=1);

namespace SignocoreToolkit\Application;

use stdClass;
use SignocoreToolkit\Infrastructure\Traits\Singleton;

final class Updater
{
	use Singleton;

	public ?string $pluginSlug = null;

	public ?string $version = null;

	private string $cacheKey = Constants::OPTION_PREFIX . 'plugin_update';

	private bool $cacheAllowed = true;

	private bool $forceCheck = false;

	protected function init(): void
	{
		if (!Constants::$isAdmin) {
			return;
		}

		$forceCheckUpdates = sanitize_text_field(wp_unslash($_GET['force-check'] ?? ''));
		if (!empty($forceCheckUpdates)) {
			$this->forceCheck = true;
			$this->cacheAllowed = false;
		}

		add_filter('plugins_api', [$this, 'info'], 20, 3);
		add_filter('pre_set_site_transient_update_plugins', [$this, 'update']);
		add_action('upgrader_process_complete', [$this, 'purge'], 10, 2);
	}

	/**
	 * Request update information from remote server.
	 *
	 * @return stdClass|false Update information object or false on failure
	 */
	public function request(): stdClass|false
	{
		if (!$this->forceCheck) {
			$lastCheck = get_transient($this->cacheKey . '_last_check');
			if ($lastCheck !== false && $lastCheck > strtotime('-12 hours')) {
				$cached = get_transient($this->cacheKey);
				if ($cached !== false) {
					$body = wp_remote_retrieve_body($cached);
					if (!empty($body)) {
						try {
							return json_decode($body, false, 512, JSON_THROW_ON_ERROR);
						} catch (\JsonException $e) {
							if (defined('WP_DEBUG') && WP_DEBUG) {
								error_log(sprintf(
									'Signocore Toolkit: JSON decode error in cached update data: %s',
									$e->getMessage()
								));
							}

							delete_transient($this->cacheKey);
						}
					}
				}
			}
		}

		$remote = get_transient($this->cacheKey);
		if ($remote === false || !$this->cacheAllowed || $this->forceCheck) {
			$remote = wp_remote_get(
				Constants::UPDATE_URL,
				[
					'timeout' => 10,
					'headers' => [
						'Accept' => 'application/json'
					]
				]
			);

			if (is_wp_error($remote)) {
				if (defined('WP_DEBUG') && WP_DEBUG) {
					error_log(sprintf(
						'Signocore Toolkit: Update check failed: %s',
						$remote->get_error_message()
					));
				}

				return false;
			}

			$responseCode = wp_remote_retrieve_response_code($remote);
			if (200 !== $responseCode) {
				if (defined('WP_DEBUG') && WP_DEBUG) {
					error_log(sprintf(
						'Signocore Toolkit: Update server returned HTTP %d',
						$responseCode
					));
				}

				return false;
			}

			$body = wp_remote_retrieve_body($remote);
			if (empty($body)) {
				if (defined('WP_DEBUG') && WP_DEBUG) {
					error_log('Signocore Toolkit: Update server returned empty response');
				}

				return false;
			}

			set_transient($this->cacheKey, $remote, DAY_IN_SECONDS);
			set_transient($this->cacheKey . '_last_check', time(), DAY_IN_SECONDS);
		}

		$body = wp_remote_retrieve_body($remote);
		if (!empty($body)) {
			try {
				return json_decode($body, false, 512, JSON_THROW_ON_ERROR);
			} catch (\JsonException $e) {
				if (defined('WP_DEBUG') && WP_DEBUG) {
					error_log(sprintf(
						'Signocore Toolkit: Invalid JSON in update response: %s',
						$e->getMessage()
					));
				}

				return false;
			}
		}

		return false;
	}

	/**
	 * @param false|object|array<string, mixed> $result
	 * @param stdClass $args
	 * @return false|object|array<string, mixed>
	 */
	public function info(false|object|array $result, string $action, object $args): false|object|array
	{
		if ('plugin_information' !== $action || Constants::TEXT_DOMAIN !== ($args->slug ?? '')) {
			return $result;
		}

		$remote = $this->request();
		if (!$remote) {
			return $result;
		}

		return $this->buildPluginInfo($remote);
	}

	/**
	 * @param stdClass $remote Remote plugin data from update server
	 */
	private function buildPluginInfo(stdClass $remote, bool $isUpdate = false): stdClass
	{
		$result = new stdClass();
		$result->name = $remote->name;
		$result->slug = $remote->slug;

		if ($isUpdate) {
			$result->new_version = $remote->version;
			$result->package = $remote->download_url;
		} else {
			$result->version = $remote->version;
		}

		$result->tested = $remote->tested;
		$result->requires = $remote->requires;
		$result->author = $remote->author;
		$result->author_profile = $remote->author_profile;
		$result->download_link = $remote->download_url;
		$result->trunk = $remote->download_url;
		$result->requires_php = $remote->requires_php;
		$result->last_updated = $remote->last_updated;
		$result->rating = $remote->rating;
		$result->num_ratings = $remote->num_ratings;
		$result->active_installs = ceil($remote->active_installs / 100) * 100;
		$result->sections = [
			'description' => $remote->sections->description,
			'installation' => $remote->sections->installation,
			'changelog' => $remote->sections->changelog
		];

		if (!empty($remote->banners)) {
			$result->banners = (array) $remote->banners;
		}

		if (!empty($remote->icons)) {
			$result->icons = (array) $remote->icons;
		}

		return $result;
	}

	/**
	 * @param stdClass|false $transient
	 * @return stdClass|false
	 */
	public function update(object|false $transient): object|false
	{
		if (empty($transient->checked)) {
			return $transient;
		}

		$remote = $this->request();
		if ($remote instanceof stdClass && !empty($remote->version) && (version_compare(Constants::$pluginVersion, $remote->version, '<') && version_compare($remote->requires, get_bloginfo('version'), '<=') && version_compare($remote->requires_php, PHP_VERSION, '<='))) {
			/** @var stdClass $transient */
			$transient->response[Constants::$pluginBase] = $this->buildPluginInfo($remote, true);
		}

		return $transient;
	}

	public function purge(mixed $upgrader, mixed $options): void
	{
		if (
			$this->cacheAllowed
			&& is_array($options)
			&& ($options['action'] ?? '') === 'update'
			&& ($options['type'] ?? '') === 'plugin'
			&& !empty($options['plugins'])
			&& in_array(Constants::$pluginBase, $options['plugins'], true)
		) {
			delete_transient($this->cacheKey);
			delete_transient($this->cacheKey . '_last_check');
			delete_site_transient('update_plugins');
		}
	}
}
