<?php

declare(strict_types=1);

namespace SignocoreToolkit\Application;

/**
 * Centralized transient management with consistent naming and expiration.
 *
 * @package SignocoreToolkit\Application
 * @since 1.0.0
 */
final class TransientManager
{
	// ==============================================
	// EXPIRATION TIME CONSTANTS
	// ==============================================

	/** @var int 1 minute */
	public const ONE_MINUTE = 60;

	/** @var int 5 minutes */
	public const FIVE_MINUTES = 300;

	/** @var int 1 hour */
	public const ONE_HOUR = 3600;

	/** @var int 12 hours */
	public const TWELVE_HOURS = 43200;

	/** @var int 1 day */
	public const ONE_DAY = 86400;

	/** @var int 1 week */
	public const ONE_WEEK = 604800;

	/** @var int 1 month */
	public const ONE_MONTH = 2592000;

	// ==============================================
	// CONTEXT CONSTANTS
	// ==============================================

	public const CONTEXT_BLOG = 'blog';
	public const CONTEXT_WOO = 'woo';
	public const CONTEXT_SOCIAL = 'social';
	public const CONTEXT_UPDATE = 'update';
	public const CONTEXT_STATS = 'stats';
	public const CONTEXT_CACHE = 'cache';

	/**
	 * Transient key prefix.
	 *
	 * @var string
	 */
	private const PREFIX = 'sctk_';

	/**
	 * Registry of transient keys created during current request.
	 *
	 * @var array<string, true>
	 */
	private static array $registry = [];

	/**
	 * Set a transient.
	 *
	 * @param string $key Transient key (without prefix)
	 * @param mixed $value Value to store
	 * @param int $expiration Expiration time in seconds
	 * @return bool True if set successfully
	 */
	public static function set(string $key, mixed $value, int $expiration): bool
	{
		$fullKey = self::buildKey($key);
		self::$registry[$fullKey] = true;

		return set_transient($fullKey, $value, $expiration);
	}

	/**
	 * Get a transient.
	 *
	 * @param string $key Transient key (without prefix)
	 * @param mixed $default Default value if not found
	 * @return mixed Transient value or default
	 */
	public static function get(string $key, mixed $default = false): mixed
	{
		$fullKey = self::buildKey($key);
		$value = get_transient($fullKey);

		return $value !== false ? $value : $default;
	}

	/**
	 * Delete a transient.
	 *
	 * @param string $key Transient key (without prefix)
	 * @return bool True if deleted successfully
	 */
	public static function delete(string $key): bool
	{
		$fullKey = self::buildKey($key);
		unset(self::$registry[$fullKey]);

		return delete_transient($fullKey);
	}

	/**
	 * Check if a transient exists and is not expired.
	 *
	 * @param string $key Transient key (without prefix)
	 * @return bool
	 */
	public static function has(string $key): bool
	{
		$fullKey = self::buildKey($key);
		return get_transient($fullKey) !== false;
	}

	/**
	 * Generate a consistent cache key with prefix.
	 *
	 * @param string $context Context identifier (blog, social, update, etc.)
	 * @param string|int|null $identifier Optional specific identifier
	 * @return string Full transient key (e.g. 'sctk_social_share_123')
	 */
	public static function key(string $context, string|int|null $identifier = null): string
	{
		$key = $context;

		if ($identifier !== null) {
			$key .= '_' . $identifier;
		}

		return self::buildKey($key);
	}

	/**
	 * Delete all transients for a specific context.
	 *
	 * @param string $context Context to delete
	 * @return int Number of transients deleted
	 */
	public static function deleteByContext(string $context): int
	{
		global $wpdb;

		$pattern = self::PREFIX . $context . '_%';
		$timeoutPattern = '_transient_timeout_' . $pattern;
		$valuePattern = '_transient_' . $pattern;

		// Delete both the transient and its timeout
		$deleted = (int) $wpdb->query(
			$wpdb->prepare(
				sprintf('DELETE FROM %s WHERE option_name LIKE %%s OR option_name LIKE %%s', $wpdb->options),
				$timeoutPattern,
				$valuePattern
			)
		);

		return $deleted;
	}

	/**
	 * Delete ALL plugin transients.
	 *
	 * @return int Number of transients deleted
	 */
	public static function deleteAll(): int
	{
		global $wpdb;

		$pattern = self::PREFIX . '%';
		$timeoutPattern = '_transient_timeout_' . $pattern;
		$valuePattern = '_transient_' . $pattern;

		// Delete both the transient and its timeout
		$deleted = (int) $wpdb->query(
			$wpdb->prepare(
				sprintf('DELETE FROM %s WHERE option_name LIKE %%s OR option_name LIKE %%s', $wpdb->options),
				$timeoutPattern,
				$valuePattern
			)
		);

		// Clear registry
		self::$registry = [];

		return $deleted;
	}

	/**
	 * Get all plugin transient keys from database.
	 *
	 * @return array<string>
	 */
	public static function getAllKeys(): array
	{
		global $wpdb;

		$pattern = '_transient_' . self::PREFIX . '%';

		$results = $wpdb->get_col(
			$wpdb->prepare(
				sprintf('SELECT option_name FROM %s WHERE option_name LIKE %%s', $wpdb->options),
				$pattern
			)
		);

		// Remove '_transient_' prefix from results
		return array_map(fn($key): string|array => str_replace('_transient_', '', $key), $results);
	}

	/**
	 * Get transient statistics for debugging.
	 *
	 * @return array<string, mixed>
	 */
	public static function getStats(): array
	{
		$keys = self::getAllKeys();
		$contexts = [];

		foreach ($keys as $key) {
			// Extract context from key (first part after prefix)
			$parts = explode('_', str_replace(self::PREFIX, '', $key));
			$context = $parts[0];

			if (!isset($contexts[$context])) {
				$contexts[$context] = 0;
			}

			$contexts[$context]++;
		}

		return [
			'total_count' => count($keys),
			'contexts' => $contexts,
			'keys' => $keys,
		];
	}

	/**
	 * Build full transient key with prefix.
	 *
	 * @param string $key Key without prefix
	 * @return string Full key with prefix
	 */
	private static function buildKey(string $key): string
	{
		// If key already has prefix, don't add it again
		if (str_starts_with($key, self::PREFIX)) {
			return $key;
		}

		return self::PREFIX . $key;
	}

	/**
	 * Refresh a transient (reset expiration).
	 *
	 * @param string $key Transient key (without prefix)
	 * @param int $expiration New expiration time in seconds
	 * @return bool True if refreshed successfully
	 */
	public static function refresh(string $key, int $expiration): bool
	{
		$value = self::get($key);

		if ($value === false) {
			return false;
		}

		return self::set($key, $value, $expiration);
	}
}
