<?php

declare(strict_types=1);

namespace SignocoreToolkit\Application;

use SignocoreToolkit\Admin\AdminTweaks;
use SignocoreToolkit\Admin\Login;
use SignocoreToolkit\Admin\SeoRecommendation;
use SignocoreToolkit\Admin\Settings;
use SignocoreToolkit\Features\Blog;
use SignocoreToolkit\Features\Cookies;
use SignocoreToolkit\Features\Files;
use SignocoreToolkit\Features\Images;
use SignocoreToolkit\Features\Kadence;
use SignocoreToolkit\Features\Mail;
use SignocoreToolkit\Features\Meta;
use SignocoreToolkit\Features\Performance;
use SignocoreToolkit\Features\Security;
use SignocoreToolkit\Features\Shortcodes;
use SignocoreToolkit\Features\Social;
use SignocoreToolkit\Features\WooCommerce;
use SignocoreToolkit\Features\DevTools\AdminBarStats;
use SignocoreToolkit\Features\DevTools\AdminMenu as DevToolsMenu;
use SignocoreToolkit\Features\DevTools\CronViewer;
use SignocoreToolkit\Features\DevTools\Database as DevToolsDatabase;
use SignocoreToolkit\Features\DevTools\Debugging;
use SignocoreToolkit\Features\DevTools\EnvironmentBanner;
use SignocoreToolkit\Features\DevTools\MailCatcher;
use SignocoreToolkit\Features\DevTools\PluginDownload;
use SignocoreToolkit\Features\DevTools\SystemStatus;
use SignocoreToolkit\Features\DevTools\ThemeDownload;
use SignocoreToolkit\Features\DevTools\TransientViewer;
use SignocoreToolkit\Features\DevTools\UserSwitcher;
use SignocoreToolkit\UI\Assets;

/**
 * Service initialization orchestrator for all plugin features and components.
 *
 * @package SignocoreToolkit\Application
 * @since 1.0.0
 */
final class ServiceRegistry
{
    /**
     * Registry of loaded services.
     *
     * @var array<string, bool>
     */
    private static array $loadedServices = [];

    /**
     * Boot all plugin services in dependency order.
     */
    public static function boot(): void
    {
        // ==========================================
        // PHASE 1: Core Infrastructure
        // ==========================================
        // Must load first as other services may depend on update info

        self::loadService(Updater::class);

        // ==========================================
        // PHASE 2: Admin Layer
        // ==========================================
        // Settings must load before features that need to read settings

        self::loadService(Login::class);
        self::loadService(AdminTweaks::class);
        self::loadService(Settings::class);

        if (Constants::$isAdmin) {
            self::loadService(SeoRecommendation::class);
        }

        // ==========================================
        // PHASE 3: UI Layer
        // ==========================================
        // Asset management for frontend resources

        self::loadService(Assets::class);

        // ==========================================
        // PHASE 4: Feature Layer
        // ==========================================
        // Features are generally independent but order matters for some

        // Integration features (load early for other features to detect)
        self::loadService(WooCommerce::class);    // E-commerce integration
        self::loadService(Kadence::class);        // Theme integration

        // Content features
        self::loadService(Cookies::class);        // Cookie consent
        self::loadService(Blog::class);           // Blog enhancements
        self::loadService(Shortcodes::class);     // Custom shortcodes
        self::loadService(Social::class);         // Social sharing
        self::loadService(Meta::class);           // Meta tags

        // File & media features
        self::loadService(Files::class);          // File handling
        self::loadService(Images::class);         // Image optimization

        // Communication features
        self::loadService(Mail::class);           // Email (uses Shortcodes)

        // Optimization features (load after content features)
        self::loadService(Performance::class);    // Performance tweaks
        self::loadService(Security::class);       // Security hardening (load last)

        // ==========================================
        // PHASE 5: Developer Tools
        // ==========================================
        // Merged from Signocore Toolkit plugin

        // Admin-only tools (menus, downloads, viewers)
        if (Constants::$isAdmin) {
            self::loadService(DevToolsDatabase::class);  // DB migrations (early)
            self::loadService(PluginDownload::class);    // Plugin zip downloads
            self::loadService(ThemeDownload::class);     // Theme zip downloads
            self::loadService(DevToolsMenu::class);      // Submenu registration
            self::loadService(TransientViewer::class);   // Transient browser
            self::loadService(CronViewer::class);        // Cron event viewer
            self::loadService(SystemStatus::class);      // System status page
        }

        // Global features (admin + frontend)
        self::loadService(Debugging::class);             // Debug mode notices
        self::loadService(EnvironmentBanner::class);     // Environment indicator
        self::loadService(MailCatcher::class);           // Email interception
        self::loadService(UserSwitcher::class);          // User impersonation
        self::loadService(AdminBarStats::class);         // Performance stats
    }

    /**
     * Load a service and track it.
     *
     * @param class-string $serviceClass Fully qualified class name
     */
    private static function loadService(string $serviceClass): void
    {
        $serviceClass::load();
        self::$loadedServices[$serviceClass] = true;
    }

    /**
     * Check if a service has been loaded.
     *
     * @param class-string $serviceClass Fully qualified class name
     * @return bool True if service has been loaded
     */
    public static function isServiceLoaded(string $serviceClass): bool
    {
        return isset(self::$loadedServices[$serviceClass]);
    }

    /**
     * Get all loaded service class names.
     *
     * @return array<string>
     */
    public static function getLoadedServices(): array
    {
        return array_keys(self::$loadedServices);
    }
}
