<?php

declare(strict_types=1);

namespace SignocoreToolkit\Application;

/**
 * Centralized type-safe options management with versioning and migration support.
 *
 * @package SignocoreToolkit\Application
 * @since 1.0.0
 */
final class OptionsManager
{
	/**
	 * Option version key for managing migrations.
	 *
	 * @var string
	 */
	public const OPTIONS_VERSION_KEY = 'options_version';

	/**
	 * Current options schema version.
	 *
	 * @var string
	 */
	public const CURRENT_VERSION = '2.0.0';

	/**
	 * Default option values (keys without OPTION_PREFIX).
	 *
	 * @var array<string, mixed>
	 */
	private const DEFAULT_OPTIONS = [
		// Cookie consent settings (GDPR-compliant)
		'add_cookie_notice' => true,
		'cookie_banner_position' => 'bottom-right',
		'cookie_banner_text' => '',
		'cookie_expiry_days' => 365,
		'cookie_show_floating_button' => true,

		// Blog settings
		'default_featured_image' => '',
		'show_related_questions' => true,
		'show_related_glossary_terms' => true,

		// Social settings
		'social_share_locations' => ['post'],

		// WooCommerce settings
		'show_safe_checkout' => 'after_summary',
		'show_ordering' => true,
		'show_result_count' => true,
		'show_related_questions_products' => true,
		'show_related_glossary_terms_products' => true,

		// Redirect settings
		'redirects_separate_menu' => true,
	];

	/**
	 * Get option value with default.
	 *
	 * @param string $key Option key (without prefix)
	 * @param mixed $default Optional default value (overrides DEFAULT_OPTIONS)
	 * @return mixed Option value or default
	 */
	public static function get(string $key, mixed $default = null): mixed
	{
		$defaultValue = $default ?? (self::DEFAULT_OPTIONS[$key] ?? null);
		$value = get_option(self::prefixKey($key), null);

		// Fallback to old prefix for backward compatibility
		if ($value === null) {
			$value = get_option('wpswpo_' . $key, null);
		}

		return $value ?? $defaultValue;
	}

	/**
	 * Get option as boolean.
	 *
	 * @param string $key Option key (without prefix)
	 * @param bool $default Default value
	 * @return bool
	 */
	public static function getBool(string $key, bool $default = false): bool
	{
		return (bool) self::get($key, $default);
	}

	/**
	 * Get option as string.
	 *
	 * @param string $key Option key (without prefix)
	 * @param string $default Default value
	 * @return string
	 */
	public static function getString(string $key, string $default = ''): string
	{
		return (string) self::get($key, $default);
	}

	/**
	 * Get option as integer.
	 *
	 * @param string $key Option key (without prefix)
	 * @param int $default Default value
	 * @return int
	 */
	public static function getInt(string $key, int $default = 0): int
	{
		return (int) self::get($key, $default);
	}

	/**
	 * Get option as array.
	 *
	 * @param string $key Option key (without prefix)
	 * @param array<mixed> $default Default value
	 * @return array<mixed>
	 */
	public static function getArray(string $key, array $default = []): array
	{
		return (array) self::get($key, $default);
	}

	/**
	 * Set option value.
	 *
	 * @param string $key Option key (without prefix)
	 * @param mixed $value Option value
	 * @return bool True if updated successfully
	 */
	public static function set(string $key, mixed $value): bool
	{
		return update_option(self::prefixKey($key), $value);
	}

	/**
	 * Delete option.
	 *
	 * @param string $key Option key (without prefix)
	 * @return bool True if deleted successfully
	 */
	public static function delete(string $key): bool
	{
		return delete_option(self::prefixKey($key));
	}

	/**
	 * Check if option exists in database.
	 *
	 * @param string $key Option key (without prefix)
	 * @return bool True if option exists
	 */
	public static function has(string $key): bool
	{
		$exists = get_option(self::prefixKey($key));
		if ($exists !== false) {
			return true;
		}
		return get_option('wpswpo_' . $key) !== false;
	}

	/**
	 * Get multiple options at once.
	 *
	 * @param array<string> $keys Array of option keys (without prefix)
	 * @return array<string, mixed>
	 */
	public static function getMultiple(array $keys): array
	{
		$options = [];

		foreach ($keys as $key) {
			$options[$key] = self::get($key);
		}

		return $options;
	}

	/**
	 * Set multiple options at once.
	 *
	 * @param array<string, mixed> $options Associative array of option key => value
	 * @return bool True if all options were updated successfully
	 */
	public static function setMultiple(array $options): bool
	{
		$success = true;

		foreach ($options as $key => $value) {
			if (!self::set($key, $value)) {
				$success = false;
			}
		}

		return $success;
	}

	/**
	 * Get all plugin options.
	 *
	 * @return array<string, mixed>
	 */
	public static function getAll(): array
	{
		$allOptions = [];

		foreach (array_keys(self::DEFAULT_OPTIONS) as $key) {
			$allOptions[$key] = self::get($key);
		}

		return $allOptions;
	}

	/**
	 * Reset option to default value.
	 *
	 * @param string $key Option key (without prefix)
	 * @param bool $explicit If true, sets to default value instead of deleting
	 * @return bool True if reset successfully
	 */
	public static function reset(string $key, bool $explicit = false): bool
	{
		if ($explicit && isset(self::DEFAULT_OPTIONS[$key])) {
			return self::set($key, self::DEFAULT_OPTIONS[$key]);
		}

		return self::delete($key);
	}

	/**
	 * Reset all options to defaults.
	 *
	 * @return bool True if all options were reset successfully
	 */
	public static function resetAll(): bool
	{
		$success = true;

		foreach (array_keys(self::DEFAULT_OPTIONS) as $key) {
			if (!self::delete($key)) {
				$success = false;
			}
		}

		return $success;
	}

	/**
	 * Get current options version.
	 *
	 * @return string
	 */
	public static function getVersion(): string
	{
		return self::getString(self::OPTIONS_VERSION_KEY, '0.0.0');
	}

	/**
	 * Update options version.
	 *
	 * @param string $version Version to set
	 * @return bool True if updated successfully
	 */
	public static function setVersion(string $version): bool
	{
		return self::set(self::OPTIONS_VERSION_KEY, $version);
	}

	/**
     * Run option migrations.
     *
     * Safe to call multiple times - migrations only run once.
     */
    public static function migrateOptions(): void
	{
		$currentVersion = self::getVersion();

		// Fresh install - set version and exit
		if ($currentVersion === '0.0.0') {
			self::setVersion(self::CURRENT_VERSION);
			return;
		}

		// Already on current version - no migration needed
		if (version_compare($currentVersion, self::CURRENT_VERSION, '>=')) {
			return;
		}

		// Migration: 1.0.0 -> 2.0.0 - Migrate wpswpo_ options to sctk_
		if (version_compare($currentVersion, '2.0.0', '<')) {
			self::migrateFromOldPrefix();
		}

		// Example migration: 0.0.0 -> 1.0.0
		// if (version_compare($currentVersion, '1.0.0', '<')) {
		//     // Migration code here
		//     // Example: Rename old option to new option
		//     // $oldValue = self::get('old_key');
		//     // self::set('new_key', $oldValue);
		//     // self::delete('old_key');
		// }

		// Update to current version after all migrations
		self::setVersion(self::CURRENT_VERSION);
	}

	/**
	 * Migrate options from old wpswpo_ prefix to sctk_.
	 */
	private static function migrateFromOldPrefix(): void
	{
		foreach (array_keys(self::DEFAULT_OPTIONS) as $key) {
			$oldValue = get_option('wpswpo_' . $key);
			if ($oldValue !== false) {
				update_option(self::prefixKey($key), $oldValue);
				delete_option('wpswpo_' . $key);
			}
		}

		// Also migrate the options version key
		$oldVersion = get_option('wpswpo_' . self::OPTIONS_VERSION_KEY);
		if ($oldVersion !== false) {
			update_option(self::prefixKey(self::OPTIONS_VERSION_KEY), $oldVersion);
			delete_option('wpswpo_' . self::OPTIONS_VERSION_KEY);
		}
	}

	/**
	 * Add option key prefix.
	 *
	 * @param string $key Option key without prefix
	 * @return string Full option key with prefix
	 */
	private static function prefixKey(string $key): string
	{
		return Constants::OPTION_PREFIX . $key;
	}

	/**
	 * Get default value for an option.
	 *
	 * @param string $key Option key (without prefix)
	 * @return mixed Default value or null
	 */
	public static function getDefault(string $key): mixed
	{
		return self::DEFAULT_OPTIONS[$key] ?? null;
	}

	/**
	 * Check if option has a defined default.
	 *
	 * @param string $key Option key (without prefix)
	 * @return bool
	 */
	public static function hasDefault(string $key): bool
	{
		return isset(self::DEFAULT_OPTIONS[$key]);
	}
}
