<?php

declare(strict_types=1);

namespace SignocoreToolkit\Application;

/**
 * Plugin lifecycle management: activation, deactivation, upgrades, and uninstallation.
 *
 * @package SignocoreToolkit\Application
 * @since 1.0.0
 */
final class Lifecycle
{
	/** Option key for storing activation timestamp */
	private const ACTIVATION_TIME_KEY = 'activation_time';

	/** Option key for storing installed version */
	private const INSTALLED_VERSION_KEY = 'installed_version';

	/**
     * Plugin activation handler.
     *
     * @param bool $networkWide Whether this is a network-wide activation
     */
    public static function activate(bool $networkWide = false): void
	{
		if ($networkWide && function_exists('is_multisite') && is_multisite()) {
			self::networkActivate();
			return;
		}

		self::singleActivate();
	}

	/**
     * Plugin deactivation handler.
     *
     * @param bool $networkWide Whether this is a network-wide deactivation
     */
    public static function deactivate(bool $networkWide = false): void
	{
		if ($networkWide && function_exists('is_multisite') && is_multisite()) {
			self::networkDeactivate();
			return;
		}

		self::singleDeactivate();
	}

	/**
     * Check and run upgrade tasks if version changed.
     */
    public static function checkUpgrade(): void
	{
		$installedVersion = OptionsManager::getString(self::INSTALLED_VERSION_KEY, '0.0.0');
		$currentVersion = Constants::$pluginVersion;

		if ($installedVersion === $currentVersion) {
			return;
		}

		self::upgrade($installedVersion, $currentVersion);

		OptionsManager::set(self::INSTALLED_VERSION_KEY, $currentVersion);
	}

	/**
     * Single site activation.
     */
    private static function singleActivate(): void
	{
		if (!OptionsManager::has(self::ACTIVATION_TIME_KEY)) {
			OptionsManager::set(self::ACTIVATION_TIME_KEY, time());
		}

		if (!OptionsManager::has(self::INSTALLED_VERSION_KEY)) {
			OptionsManager::set(self::INSTALLED_VERSION_KEY, Constants::$pluginVersion);
		}

		OptionsManager::migrateOptions();

		flush_rewrite_rules();

		TransientManager::deleteAll();

		if (defined('WP_DEBUG') && WP_DEBUG) {
			error_log('Signocore Toolkit activated (version ' . Constants::$pluginVersion . ')');
		}

		do_action('signocore_toolkit_activated');
	}

	/**
     * Single site deactivation (does not delete user data).
     */
    private static function singleDeactivate(): void
	{
		TransientManager::deleteAll();

		flush_rewrite_rules();

		self::clearScheduledEvents();

		if (defined('WP_DEBUG') && WP_DEBUG) {
			error_log('Signocore Toolkit deactivated');
		}

		do_action('signocore_toolkit_deactivated');
	}

	/**
     * Network-wide activation.
     */
    private static function networkActivate(): void
	{
		if (!function_exists('get_sites')) {
			return;
		}

		$sites = get_sites(['number' => 0]);

		foreach ($sites as $site) {
			switch_to_blog((int) $site->blog_id);
			self::singleActivate();
			restore_current_blog();
		}
	}

	/**
     * Network-wide deactivation.
     */
    private static function networkDeactivate(): void
	{
		if (!function_exists('get_sites')) {
			return;
		}

		$sites = get_sites(['number' => 0]);

		foreach ($sites as $site) {
			switch_to_blog((int) $site->blog_id);
			self::singleDeactivate();
			restore_current_blog();
		}
	}

	/**
     * Handle plugin version upgrade.
     *
     * @param string $oldVersion Previous version
     * @param string $newVersion New version
     */
    private static function upgrade(string $oldVersion, string $newVersion): void
	{
		OptionsManager::migrateOptions();

		flush_rewrite_rules();

		if (defined('WP_DEBUG') && WP_DEBUG) {
			error_log(sprintf('Signocore Toolkit upgraded from %s to %s', $oldVersion, $newVersion));
		}

		do_action('signocore_toolkit_upgraded', $oldVersion, $newVersion);
	}

	/**
     * Complete uninstallation - removes all plugin data.
     */
    public static function uninstall(): void
	{
		OptionsManager::resetAll();
		OptionsManager::delete(self::ACTIVATION_TIME_KEY);
		OptionsManager::delete(self::INSTALLED_VERSION_KEY);
		OptionsManager::delete(OptionsManager::OPTIONS_VERSION_KEY);

		TransientManager::deleteAll();

		// Clean up toolkit database tables
		global $wpdb;
		$wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}sctk_caught_emails");

		// Clean up toolkit options
		delete_option('sctk_environment_override');
		delete_option('sctk_mail_log_mode');
		delete_option('sctk_mail_prevent_sending');
		delete_option('sctk_mail_auto_delete_days');
		delete_option('sctk_dev_db_version');
		delete_option('swift_dev_db_version'); // Legacy cleanup

		// Clean up toolkit user meta
		$wpdb->delete($wpdb->usermeta, ['meta_key' => 'sctk_original_user_id']);

		self::clearScheduledEvents();

		flush_rewrite_rules();

		if (defined('WP_DEBUG') && WP_DEBUG) {
			error_log('Signocore Toolkit uninstalled - all data removed');
		}

		do_action('signocore_toolkit_uninstalled');
	}

	/**
     * Clear all scheduled cron events.
     */
    private static function clearScheduledEvents(): void
	{
		// If plugin schedules any cron events, clear them here
		// Example:
		// wp_clear_scheduled_hook('signocore_toolkit_daily_cleanup');
		// wp_clear_scheduled_hook('signocore_toolkit_weekly_check');

		// Currently, this plugin doesn't use scheduled events
	}

	/**
	 * Get activation timestamp.
	 *
	 * @return int|null Activation timestamp or null if never activated
	 */
	public static function getActivationTime(): ?int
	{
		$time = OptionsManager::getInt(self::ACTIVATION_TIME_KEY, 0);
		return $time > 0 ? $time : null;
	}

	/**
	 * Get installed version.
	 *
	 * @return string Installed version number
	 */
	public static function getInstalledVersion(): string
	{
		return OptionsManager::getString(self::INSTALLED_VERSION_KEY, '0.0.0');
	}

	/**
	 * Check if this is a fresh installation.
	 *
	 * @return bool True if fresh install
	 */
	public static function isFreshInstall(): bool
	{
		return !OptionsManager::has(self::ACTIVATION_TIME_KEY);
	}

	/**
	 * Get number of days since plugin was first activated.
	 *
	 * @return int Days since activation (0 if never activated)
	 */
	public static function getDaysSinceActivation(): int
	{
		$activationTime = self::getActivationTime();

		if ($activationTime === null) {
			return 0;
		}

		return (int) floor((time() - $activationTime) / DAY_IN_SECONDS);
	}
}
