<?php

declare(strict_types=1);

namespace SignocoreToolkit\Application;

use SignocoreToolkit\Support\UrlHelper;

/**
 * Central constants and runtime configuration for the plugin.
 *
 * @package SignocoreToolkit\Application
 * @since 1.0.0
 */
final class Constants
{
    // ==========================================
    // TRUE CONSTANTS (Compile-time)
    // ==========================================

    /** @var string Database option prefix */
    public const OPTION_PREFIX = 'sctk_';

    /** @var string Translation text domain */
    public const TEXT_DOMAIN = 'signocore-toolkit';

    /** @var string Custom update server URL */
    public const UPDATE_URL = 'https://update.signocore.com/api/wp-update/plugins/' . self::TEXT_DOMAIN;

    /** @var string Absolute path to templates directory */
    public const TEMPLATE_DIR = SIGNOCORE_TOOLKIT_DIR . 'templates/';

    // ==========================================
    // RUNTIME PROPERTIES (Initialized via load())
    // ==========================================

    /** Plugin version number extracted from the main plugin file header. */
    public static string $pluginVersion;

    /** Plugin basename for WordPress hooks and actions. */
    public static string $pluginBase;

    /** True if site is visible to search engines. */
    public static bool $isPublic;

    /** True if current request is in WordPress admin area. */
    public static bool $isAdmin;

    /** Full URL to assets directory. */
    public static string $assetsUri;

    /** True if WooCommerce plugin is active. */
    public static bool $isWooCommerce;

    /** Fully sanitized URL of the current page. */
    public static string $currentUrl;

    /**
     * Initialize all runtime properties.
     *
     * Must be called during 'plugins_loaded' or later.
     */
    public static function load(): void
    {
        self::$pluginVersion = get_file_data(SIGNOCORE_TOOLKIT_FILE, ['Version'])[0] ?? '';
        self::$pluginBase = plugin_basename(SIGNOCORE_TOOLKIT_FILE);
        self::$isPublic = (bool) get_option('blog_public', false);
        self::$isAdmin = is_admin();
        self::$assetsUri = plugin_dir_url(SIGNOCORE_TOOLKIT_FILE) . 'assets';
        self::$isWooCommerce = class_exists('WooCommerce');
        self::$currentUrl = UrlHelper::currentUrl();
    }

    /**
     * Safely retrieve a runtime property with a fallback value.
     *
     * @param string $property Property name (without $)
     * @param mixed $default Default value if property is not initialized
     * @return mixed Property value or default
     */
    public static function get(string $property, mixed $default = null): mixed
    {
        return self::${$property} ?? $default;
    }
}
