<?php

declare(strict_types=1);

namespace SignocoreToolkit\Admin;

use SignocoreToolkit\Infrastructure\Traits\Singleton;
use SignocoreToolkit\Application\Constants;
use SignocoreToolkit\Support\PostTypeHelper;
use SignocoreToolkit\Support\UrlHelper;

final class Settings
{
	use Singleton;

	/** @phpstan-ignore property.onlyWritten */
	private static bool $isResolvingOption = false;

	public ?string $optionSuffix = null;

	/** @var array<string, array<string, mixed>> */
	public array $settings = [];

	public mixed $postTypes = null;

	protected function init(): void
	{
		add_action('admin_menu', [$this, 'addSettingsMenus']);
		add_action('admin_init', [$this, 'registerSettings']);
		add_action('admin_notices', [$this, 'showUpdatedNotice']);

		add_action('admin_enqueue_scripts', [$this, 'enqueueScripts']);
	}

	private function ensureSettings(): void
	{
		if ($this->settings === []) {
			$this->settings = $this->getSettings();
		}
	}

	public function enqueueScripts(): void
	{
		$screen = get_current_screen();
		if (!$screen || strpos($screen->id, 'signocore-toolkit') === false) {
			return;
		}

		$assetHandle = Constants::TEXT_DOMAIN . '-a';

		wp_enqueue_style($assetHandle, Constants::$assetsUri . '/css/admin.min.css', [], Constants::$pluginVersion);
		wp_enqueue_script($assetHandle, Constants::$assetsUri . '/js/admin.min.js', ['jquery'], Constants::$pluginVersion, true);
	}

	/** @deprecated Use ensureSettings() instead - kept for backwards compatibility */
	public function setupProperties(): void
	{
		$this->ensureSettings();
	}

	/**
     * @return array<string, array<string, mixed>>
     */
    public function getSettings(): array
	{
		// Init array
		$settings = [];

		// General settings
		$settings['signocore-toolkit-settings'] = [
			'full_title' => __('General Settings', Constants::TEXT_DOMAIN),
			'menu_title' => __('Signocore Toolkit', Constants::TEXT_DOMAIN),
			'tab_title' => __('General', Constants::TEXT_DOMAIN),
			'groups' => [
				'cookies' => [
					'title' => __('GDPR Cookie Consent', Constants::TEXT_DOMAIN),
					'fields' => [
						[
							'id' => 'add_cookie_notice',
							'type' => 'checkbox',
							'label' => __('Enable Cookie Consent', Constants::TEXT_DOMAIN),
							'description' => __('Show GDPR-compliant cookie consent banner with granular category controls.', Constants::TEXT_DOMAIN),
							'default' => true,
						],
						[
							'id' => 'cookie_banner_position',
							'type' => 'select',
							'label' => __('Banner Position', Constants::TEXT_DOMAIN),
							'description' => __('Where to display the consent banner on the page.', Constants::TEXT_DOMAIN),
							'options' => [
								'bottom-right' => __('Bottom Right', Constants::TEXT_DOMAIN),
								'bottom-left' => __('Bottom Left', Constants::TEXT_DOMAIN),
								'bottom-full' => __('Full Width Bottom', Constants::TEXT_DOMAIN),
							],
							'default' => 'bottom-right',
							'with_keys' => true,
						],
						[
							'id' => 'cookie_banner_text',
							'type' => 'textarea',
							'label' => __('Banner Message', Constants::TEXT_DOMAIN),
							'description' => __('Custom message for the consent banner. Leave empty to use the default GDPR-compliant text.', Constants::TEXT_DOMAIN),
							'default' => '',
						],
						[
							'id' => 'cookie_expiry_days',
							'type' => 'number',
							'label' => __('Consent Duration (Days)', Constants::TEXT_DOMAIN),
							'description' => __('How long to remember user consent choices. GDPR recommends reviewing consent annually (365 days).', Constants::TEXT_DOMAIN),
							'default' => 365,
							'min' => '30',
							'max' => '730',
						],
						[
							'id' => 'cookie_show_floating_button',
							'type' => 'checkbox',
							'label' => __('Show Preferences Button', Constants::TEXT_DOMAIN),
							'description' => __('Display a floating button allowing users to change their consent at any time (required by GDPR).', Constants::TEXT_DOMAIN),
							'default' => true,
						],
						[
							'id' => 'cookie_gdpr_info',
							'type' => 'message',
							'label' => '',
							'description' => '<span class="dashicons dashicons-yes-alt" style="color: #00a32a"></span> ' . __('This cookie consent system is GDPR-compliant with: granular consent per category (Necessary, Analytics, Marketing), default deny for non-essential cookies, and the ability for users to withdraw consent at any time.', Constants::TEXT_DOMAIN),
							'default' => '',
						],
					],
				],
			]
		];

		// Blog settings
		$settings['signocore-toolkit-blog'] = [
			'full_title' => __('Blog Settings', Constants::TEXT_DOMAIN),
			'menu_title' => __('Blog', Constants::TEXT_DOMAIN),
			'tab_title' => __('Blog', Constants::TEXT_DOMAIN),
			'groups' => [
				'images' => [
					'title' => __('Images', Constants::TEXT_DOMAIN),
					'fields' => [
						[
							'id' => 'default_featured_image',
							'type' => 'image',
							'label' => __('Default Featured Image', Constants::TEXT_DOMAIN),
							'description' => __('This image will be used as featured blog post image if no featured image was set.', Constants::TEXT_DOMAIN),
							'default' => '',
						],
					],
				],
			],
		];

		if (defined('SIGNOCORE_FAQ_VERSION') || defined('SIGNOCORE_GLOSSARY_VERSION')) {
			$settings['signocore-toolkit-blog']['groups']['related'] = [
				'title' => __('Related Content', Constants::TEXT_DOMAIN),
				'fields' => [],
			];
		}

		if (defined('SIGNOCORE_FAQ_VERSION')) {
			$settings['signocore-toolkit-blog']['groups']['related']['fields'][] = [
				'id' => 'show_related_questions',
				'type' => 'checkbox',
				'label' => __('Related FAQs', Constants::TEXT_DOMAIN),
				'description' => __('Show related FAQs', Constants::TEXT_DOMAIN),
				'default' => true,
			];
		}

		if (defined('SIGNOCORE_GLOSSARY_VERSION')) {
			$settings['signocore-toolkit-blog']['groups']['related']['fields'][] = [
				'id' => 'show_related_glossary_terms',
				'type' => 'checkbox',
				'label' => __('Related Terms', Constants::TEXT_DOMAIN),
				'description' => __('Show related Glossary terms', Constants::TEXT_DOMAIN),
				'default' => true,
			];
		}

		// Social settings
		$settings['signocore-toolkit-social'] = [
			'full_title' => __('Social Settings', Constants::TEXT_DOMAIN),
			'menu_title' => __('Social', Constants::TEXT_DOMAIN),
			'tab_title' => __('Social', Constants::TEXT_DOMAIN),
			'groups' => [
				'sharing' => [
					'title' => __('Sharing', Constants::TEXT_DOMAIN),
					'fields' => [
						[
							'id' => 'social_share_locations',
							'type' => 'checkbox_group',
							'label' => __('Add social sharing', Constants::TEXT_DOMAIN),
							'description' => __('Add social sharing on these post types', Constants::TEXT_DOMAIN),
							'options' => PostTypeHelper::getPostTypes(),
							'default' => ['post'],
						],
					],
				],
			],
		];

		// Shop settings
		if (Constants::$isWooCommerce) {
			$settings['signocore-toolkit-shop'] = [
				'full_title' => __('Shop Settings', Constants::TEXT_DOMAIN),
				'menu_title' => __('Shop', Constants::TEXT_DOMAIN),
				'tab_title' => __('Shop', Constants::TEXT_DOMAIN),
				'groups' => [
					'general' => [
						'title' => __('General', Constants::TEXT_DOMAIN),
						'fields' => [
							[
								'id' => 'show_ordering',
								'type' => 'checkbox',
								'label' => __('Ordering', Constants::TEXT_DOMAIN),
								'description' => __('Show ordering dropdown', Constants::TEXT_DOMAIN),
								'default' => true,
							],
							[
								'id' => 'show_result_count',
								'type' => 'checkbox',
								'label' => __('Result Count', Constants::TEXT_DOMAIN),
								'description' => __('Show result count on shop page', Constants::TEXT_DOMAIN),
								'default' => true,
							],
						],
					],
					'images' => [
						'title' => __('Images', Constants::TEXT_DOMAIN),
						'fields' => [
							[
								'id' => 'payment_methods',
								'type' => 'checkbox_group',
								'label' => __('Payment Logos', Constants::TEXT_DOMAIN),
								'description' => __('Choose which payment logos should be printed when using the [payment-methods] shortcode', Constants::TEXT_DOMAIN),
								'options' => [
									'alipay' => __('Alipay', Constants::TEXT_DOMAIN),
									'amex' => __('American Express', Constants::TEXT_DOMAIN),
									'apple-pay' => __('Apple Pay', Constants::TEXT_DOMAIN),
									'bank' => __('Bank Transfer', Constants::TEXT_DOMAIN),
									'bitcoin' => __('Bitcoin', Constants::TEXT_DOMAIN),
									'dankort' => __('Dankort', Constants::TEXT_DOMAIN),
									'discover' => __('Discover', Constants::TEXT_DOMAIN),
									'ean' => __('EAN', Constants::TEXT_DOMAIN),
									'google-pay' => __('Google Pay', Constants::TEXT_DOMAIN),
									'jcb' => __('JCB', Constants::TEXT_DOMAIN),
									'klarna' => __('Klarna', Constants::TEXT_DOMAIN),
									'maestro' => __('Maestro', Constants::TEXT_DOMAIN),
									'master' => __('MasterCard', Constants::TEXT_DOMAIN),
									'mobilepay' => __('MobilePay', Constants::TEXT_DOMAIN),
									'paypal' => __('PayPal', Constants::TEXT_DOMAIN),
									'resurs' => __('Resurs', Constants::TEXT_DOMAIN),
									'santander' => __('Santander', Constants::TEXT_DOMAIN),
									'sparxpres' => __('Sparxpres', Constants::TEXT_DOMAIN),
									'swish' => __('Swish', Constants::TEXT_DOMAIN),
									'unionpay' => __('Unionpay', Constants::TEXT_DOMAIN),
									'viabill' => __('Viabill', Constants::TEXT_DOMAIN),
									'visa-electron' => __('Visa Electron', Constants::TEXT_DOMAIN),
									'visa' => __('Visa', Constants::TEXT_DOMAIN),
								],
								'default' => ['visa', 'master'],
							],
							[
								'id' => 'show_safe_checkout',
								'type' => 'radio',
								'label' => __('Safe Checkout', Constants::TEXT_DOMAIN),
								'description' => __('Show safe checkout', Constants::TEXT_DOMAIN),
								'options' => [
									'none' => __('Do not show', Constants::TEXT_DOMAIN),
									'before_quantity' => __('Before quantity', Constants::TEXT_DOMAIN),
									'after_summary' => __('After summary', Constants::TEXT_DOMAIN),
								],
								'default' => 'after_summary',
							],
						],
					],
				],
			];

			if (defined('SIGNOCORE_FAQ_VERSION') || defined('SIGNOCORE_GLOSSARY_VERSION')) {
				$settings['signocore-toolkit-shop']['groups']['related'] = [
					'title' => __('Related Content', Constants::TEXT_DOMAIN),
					'fields' => [],
				];
			}

			if (defined('SIGNOCORE_FAQ_VERSION')) {
				$settings['signocore-toolkit-shop']['groups']['related']['fields'][] = [
					'id' => 'show_related_questions_products',
					'type' => 'checkbox',
					'label' => __('Related FAQs', Constants::TEXT_DOMAIN),
					'description' => __('Show related FAQs', Constants::TEXT_DOMAIN),
					'default' => true,
				];
			}

			if (defined('SIGNOCORE_GLOSSARY_VERSION')) {
				$settings['signocore-toolkit-shop']['groups']['related']['fields'][] = [
					'id' => 'show_related_glossary_terms_products',
					'type' => 'checkbox',
					'label' => __('Related Terms', Constants::TEXT_DOMAIN),
					'description' => __('Show related Glossary terms', Constants::TEXT_DOMAIN),
					'default' => true,
				];
			}
		}

		// Dev Tools settings
		$settings['signocore-toolkit-devtools'] = [
			'full_title' => __('Developer Tools', Constants::TEXT_DOMAIN),
			'menu_title' => __('Dev Tools', Constants::TEXT_DOMAIN),
			'tab_title' => __('Dev Tools', Constants::TEXT_DOMAIN),
			'groups' => [
				'environment' => [
					'title' => __('Environment', Constants::TEXT_DOMAIN),
					'fields' => [
						[
							'id' => 'sctk_environment_override',
							'raw_id' => true,
							'type' => 'select',
							'label' => __('Environment Override', Constants::TEXT_DOMAIN),
							'description' => __('Override the auto-detected environment type.', Constants::TEXT_DOMAIN),
							'options' => [
								'auto' => __('Auto-detect', Constants::TEXT_DOMAIN),
								'development' => __('Development', Constants::TEXT_DOMAIN),
								'staging' => __('Staging', Constants::TEXT_DOMAIN),
								'production' => __('Production', Constants::TEXT_DOMAIN),
							],
							'default' => 'auto',
							'with_keys' => true,
						],
					],
				],
				'mail-log' => [
					'title' => __('Mail Log', Constants::TEXT_DOMAIN),
					'fields' => [
						[
							'id' => 'sctk_mail_log_mode',
							'raw_id' => true,
							'type' => 'select',
							'label' => __('Log Mode', Constants::TEXT_DOMAIN),
							'description' => __('Choose whether to log all emails or only blocked ones.', Constants::TEXT_DOMAIN),
							'options' => [
								'all' => __('All emails', Constants::TEXT_DOMAIN),
								'blocked' => __('Blocked emails only', Constants::TEXT_DOMAIN),
							],
							'default' => 'all',
							'with_keys' => true,
						],
						[
							'id' => 'sctk_mail_prevent_sending',
							'raw_id' => true,
							'type' => 'checkbox',
							'label' => __('Prevent Sending', Constants::TEXT_DOMAIN),
							'description' => __('Block outgoing emails in non-production environments. Emails will be caught but not actually sent (except in production).', Constants::TEXT_DOMAIN),
							'default' => false,
						],
						[
							'id' => 'sctk_mail_auto_delete_days',
							'raw_id' => true,
							'type' => 'number',
							'label' => __('Auto-delete After', Constants::TEXT_DOMAIN),
							'description' => __('Automatically delete caught emails older than this many days. Set to 0 to never auto-delete.', Constants::TEXT_DOMAIN),
							'default' => 30,
							'min' => '0',
							'step' => '1',
						],
					],
				],
			],
		];

		return $settings;
	}

	public function registerSettings(): void
	{
		$this->ensureSettings();
		$this->registerSettingsFields($this->settings);
	}

	public function addSettingsMenus(): void
	{
		$this->ensureSettings();
		$counter = 1;
		$mainSettings = '';

		foreach ($this->settings as $key => $data) {
			$fullTitle = $data['full_title'];
			$menuTitle = $data['menu_title'];
			$subMenuTitle = $data['tab_title'] ?? $menuTitle;

			if ($counter === 1) {
				$mainSettings = $key;
				$iconPath = plugin_dir_path(SIGNOCORE_TOOLKIT_FILE) . 'assets/img/signocore.svg';
				$iconContent = file_exists($iconPath) ? file_get_contents($iconPath) : false;
				$menuIcon = $iconContent !== false
				? 'data:image/svg+xml;base64,' . base64_encode($iconContent)
				: 'dashicons-performance';
				add_menu_page($fullTitle, $menuTitle, 'manage_options', $key, [$this, 'renderSettingsPage'], $menuIcon, 96);
				add_submenu_page($key, $fullTitle, $subMenuTitle, 'manage_options', $key, [$this, 'renderSettingsPage']);
			} else {
				add_submenu_page($mainSettings, $fullTitle, $menuTitle, 'manage_options', $key, [$this, 'renderSettingsPage']);
			}

			++$counter;
		}
	}

	public function renderSettingsPage(): void
	{
		$currentPage = sanitize_text_field(wp_unslash($_GET['page'] ?? ''));

		if (!empty($this->settings[$currentPage])) {
			$settings = $this->settings[$currentPage];

			echo '<div class="wrap">
				<h1>' . esc_html($settings['full_title']) . '</h1>';

			// DevTools has its own tab navigation
			if ('signocore-toolkit-devtools' === $currentPage) {
				$this->addDevToolsTabs($currentPage);
			} else {
				$this->addTabs($currentPage);
			}

			echo '<form method="post" action="options.php" novalidate="novalidate">';

			settings_fields($currentPage);
			do_settings_sections($currentPage);
			submit_button();

			echo '</form>
			</div>';
		}
	}

	private function addTabs(string $currentPage): void
	{
		echo '<h2 class="nav-tab-wrapper wpsseo-tabs">' . "\n";

		foreach ($this->settings as $key => $section) {
			if (empty($section['tab_title'])) {
				continue;
			}

			$url = admin_url('/admin.php?page=' . $key);
			$title = $section['tab_title'];
			$active = $currentPage == $key;

			$class = 'nav-tab';
			if ($active) {
				$class .= ' nav-tab-active';
			}

			echo '<a href="' . esc_url($url) . '" class="' . esc_attr($class) . '">' . esc_html($title) . '</a>' . "\n";
		}

		echo '</h2>' . "\n";
	}

	/**
	 * Render DevTools tab navigation.
	 *
	 * Shows main plugin tabs first (with Dev Tools active),
	 * then DevTools sub-tabs with secondary styling.
	 *
	 * @param string $currentPage Current page slug.
	 */
	private function addDevToolsTabs(string $currentPage): void
	{
		// Main tabs (with Dev Tools active)
		$this->addTabs('signocore-toolkit-devtools');

		// Sub-tabs
		$tabs = [
			'signocore-toolkit-devtools' => __('Settings', Constants::TEXT_DOMAIN),
			'signocore-toolkit-mail-log' => __('Mail Log', Constants::TEXT_DOMAIN),
			'signocore-toolkit-transients' => __('Transients', Constants::TEXT_DOMAIN),
			'signocore-toolkit-cron' => __('Cron Jobs', Constants::TEXT_DOMAIN),
			'signocore-toolkit-status' => __('System Status', Constants::TEXT_DOMAIN),
		];

		echo '<div class="sctk-subtabs">';
		foreach ($tabs as $slug => $label) {
			$class = ($currentPage === $slug) ? 'sctk-subtab sctk-subtab-active' : 'sctk-subtab';
			$url = admin_url('admin.php?page=' . $slug);
			printf(
				'<a href="%s" class="%s">%s</a>',
				esc_url($url),
				esc_attr($class),
				esc_html($label)
			);
		}
		echo '</div>';
	}

	/**
	 * @param array<string, array<string, mixed>> $settings
	 */
	public function registerSettingsFields(array $settings): void
	{
		foreach ($settings as $location => $data) {
			foreach ($data['groups'] as $group) {
				$groupTitle = $group['title'];
				$groupId = sanitize_title($groupTitle);
				$descriptionCallback = $group['description'] ?? '__return_empty_string';
				$fields = $group['fields'];

				$groupClass = '';
				if (!empty($group['toggle_by'])) {
					$groupClass = ' hidden toggle-by-' . $group['toggle_by'];
				}

				add_settings_section($groupId, $groupTitle, $descriptionCallback, $location, ['before_section' => '<div class="postbox wpsseo wpsseo-settings' . $groupClass . '"><div class="inside">', 'after_section' => '</div></div>']);

				foreach ($fields as $field) {
					if (!is_array($field)) {
						continue;
					}

					if (empty($field['id'])) {
						$id = md5((empty($field['description']) ? $field['type'] : $field['description']) . microtime(true));
						$field['id'] = Constants::OPTION_PREFIX . $id;
					} elseif (!empty($field['raw_id'])) {
						// Use id as-is without prefix (for sctk_ options)
					} else {
						$field['id'] = Constants::OPTION_PREFIX . $field['id'];
					}

					if (isset($field['is_global']) && $field['is_global'] === false) {
						$field['id'] .= $this->optionSuffix;
					}

					if (!empty($field['toggle_by'])) {
						$field['class'] = 'hidden toggle-by-' . $field['toggle_by'];
					}

					register_setting($location, $field['id']);
					add_settings_field($field['id'], ($field['label'] ?? ''), [$this, 'renderSettingsField'], $location, $groupId, $field);
				}
			}
		}
	}

	/**
     * Render settings field by delegating to specific field type renderers.
     *
     * @param array<string, mixed> $field Field configuration array
     */
    public function renderSettingsField(array $field): void
	{
		$fieldType = $field['type'] ?? 'text';

		match ($fieldType) {
			'checkbox' => $this->renderCheckboxField($field),
			'checkbox_group' => $this->renderCheckboxGroupField($field),
			'image' => $this->renderImageField($field),
			'select' => $this->renderSelectField($field),
			'select_multi' => $this->renderSelectMultiField($field),
			'radio' => $this->renderRadioField($field),
			'color' => $this->renderColorField($field),
			'textarea' => $this->renderTextareaField($field),
			'number' => $this->renderNumberField($field),
			'password' => $this->renderPasswordField($field),
			'editor' => $this->renderEditorField($field),
			'message' => $this->renderMessageField($field),
			'text_group' => $this->renderTextGroupField($field),
			default => $this->renderTextField($field),
		};
	}

	public function showUpdatedNotice(): void
	{
		$this->ensureSettings();

		$settingsPages = array_keys($this->settings);
		$page = sanitize_text_field(wp_unslash($_GET['page'] ?? ''));
		if (!in_array($page, $settingsPages)) {
			return;
		}

		$updated = sanitize_text_field(wp_unslash($_GET['settings-updated'] ?? ''));
		if (empty($updated)) {
			return;
		}

		$class = 'notice notice-success is-dismissable signocore-notice';
		printf('<div class="%1$s"><p>%2$s</p></div>', esc_attr($class), esc_html__('Settings saved', Constants::TEXT_DOMAIN));
	}

	/**
     * @param array<string, mixed> $field Field configuration
     */
    private function renderCheckboxField(array $field): void
	{
		$dataAtts = '';
		if (!empty($field['toggle_target'])) {
			$dataAtts .= ' data-toggle-target=".toggle-by-' . esc_attr($field['toggle_target']) . '" data-toggle-value="1"';
		}

		echo '<fieldset>';

		$default = get_option($field['id'], $field['default']);
		$disabled = (bool) ($field['disabled'] ?? false);

		echo '<label class="wpsseo-check-switch"><input name="' . esc_attr($field['id']) . '" id="' . esc_attr($field['id']) . '" type="checkbox" value="1"' . $dataAtts . (empty($default) ? '' : ' checked') . ($disabled === false ? '' : ' disabled') . '><span class="slider"></span></label>';
		echo '<span class="toggle-label">' . wp_kses_post($field['description']) . '</span>';

		echo '</fieldset>';
	}

	/**
     * @param array<string, mixed> $field Field configuration
     */
    private function renderCheckboxGroupField(array $field): void
	{
		echo '<fieldset class="checkbox-group toggle-group' . (empty($field['inline']) ? '' : ' inline') . '">';

		if (!empty($field['options'])) {
			$default = get_option($field['id'], $field['default']);

			if (!is_array($default)) {
				$default = [$default];
			}

			$i = 0;
			foreach ($field['options'] as $key => $label) {
				$isChecked = in_array($key, $default) || !empty($default[$key]);
				echo '<div class="toggle-group-item">';
				echo '<label for="' . esc_attr($field['id']) . '-' . ++$i . '" class="wpsseo-check-switch"><input name="' . esc_attr($field['id']) . '[' . esc_attr($key) . ']" id="' . esc_attr($field['id']) . '-' . $i . '" type="checkbox" value="1"' . ($isChecked ? ' checked' : '') . '><span class="slider"></span></label>';
				echo '<span class="switch-label">' . esc_html($label) . '</span>';
				echo '</div>';
			}
		}

		if (!empty($field['description'])) {
			echo '<p class="description" id="' . esc_attr($field['id']) . '-description">' . wp_kses_post($field['description']) . '</p>';
		}

		echo '</fieldset>';
	}

	/**
     * @param array<string, mixed> $field Field configuration
     */
    private function renderImageField(array $field): void
	{
		$imageId = get_option($field['id']);
		$imageObject = '';

		if (!empty($imageId)) {
			$imageObject = wp_get_attachment_image($imageId, 'medium');
		}

		echo '<input name="' . esc_attr($field['id']) . '" id="' . esc_attr($field['id']) . '" type="hidden" value="' . esc_attr($imageId) . '">
		<style>#' . esc_attr($field['id']) . '-preview .image img{margin-bottom: 10px;}</style>
		<div id="' . esc_attr($field['id']) . '-preview"><div class="image">' . $imageObject . '</div></div>
		<input class="wpsseo-upload-image wpsseo-button" type="button" data-id="' . esc_attr($field['id']) . '" value="' . esc_attr__('Select image', Constants::TEXT_DOMAIN) . '">
		<input class="remove-image wpsseo-button" type="button" data-id="' . esc_attr($field['id']) . '" value="' . esc_attr__('Remove image', Constants::TEXT_DOMAIN) . '"' . (empty($imageId) ? ' style="display: none;"' : '') . '>';

		if (!empty($field['description'])) {
			echo '<p class="description" id="' . esc_attr($field['id']) . '-description">' . wp_kses_post($field['description']) . '</p>';
		}
	}

	/**
     * @param array<string, mixed> $field Field configuration
     */
    private function renderSelectField(array $field): void
	{
		$withKeys = (bool) ($field['with_keys'] ?? false);
		$dataAtts = '';
		if (!empty($field['toggle_target'])) {
			$dataAtts .= ' data-toggle-target=".toggle-by-' . esc_attr($field['toggle_target']) . '"';
		}

		if (!empty($field['toggle_value'])) {
			$dataAtts .= ' data-toggle-value="' . esc_attr($field['toggle_value']) . '"';
		}

		echo '<select name="' . esc_attr($field['id']) . '" id="' . esc_attr($field['id']) . '" class="regular-text styled-select"' . $dataAtts . '>';

		if (!empty($field['options'])) {
			$default = (string) get_option($field['id'], $field['default']);
			$includeKey = $field['include_sub_keys'] ?? false;

			foreach ($field['options'] as $key => $value) {
				if (!$withKeys && is_numeric($key)) {
					$key = $value;
				}

				if (is_array($value)) {
					echo '<optgroup label="' . esc_attr($key) . '">';

					if ($includeKey) {
						echo '<option value="' . esc_attr($key) . '"' . ((string) $key === $default ? ' selected' : '') . '>' . $key . '</option>';
					}

					foreach ($value as $subKey => $subValue) {
						if (!$withKeys && is_numeric($subKey)) {
							$subKey = $subValue;
						}

						if (is_array($subValue)) {
							foreach ($subValue as $subSubKey => $subSubValue) {
								if (!$withKeys && is_numeric($subSubKey)) {
									$subSubKey = $subSubValue;
								}

								$subSubValue = str_repeat('&nbsp;', 3) . $subSubValue;

								echo '<option value="' . esc_attr($subSubKey) . '"' . ((string) $subSubKey === $default ? ' selected' : '') . '>' . $subSubValue . '</option>';
							}
						} else {
							echo '<option value="' . esc_attr($subKey) . '"' . ((string) $subKey === $default ? ' selected' : '') . '>' . $subValue . '</option>';
						}
					}

					echo '</optgroup>';
				} else {
					echo '<option value="' . esc_attr($key) . '"' . ((string) $key === $default ? ' selected' : '') . '>' . $value . '</option>';
				}
			}
		}

		echo '</select>';

		if (!empty($field['description'])) {
			echo '<p class="description" id="' . esc_attr($field['id']) . '-description">' . wp_kses_post($field['description']) . '</p>';
		}
	}

	/**
     * @param array<string, mixed> $field Field configuration
     */
    private function renderSelectMultiField(array $field): void
	{
		echo '<select name="' . esc_attr($field['id']) . '[]" id="' . esc_attr($field['id']) . '" class="regular-text styled-select" multiple>';

		if (!empty($field['options'])) {
			$default = get_option($field['id'], $field['default']);

			if (!is_array($default)) {
				$default = [$default];
			}

			foreach ($field['options'] as $key => $value) {
				echo '<option value="' . esc_attr($key) . '"' . (in_array($key, $default) ? ' selected' : '') . '>' . esc_html($value) . '</option>';
			}
		}

		echo '</select>';

		if (!empty($field['description'])) {
			echo '<p class="description" id="' . esc_attr($field['id']) . '-description">' . wp_kses_post($field['description']) . '</p>';
		}
	}

	/**
     * @param array<string, mixed> $field Field configuration
     */
    private function renderRadioField(array $field): void
	{
		echo '<fieldset>';

		if (!empty($field['options'])) {
			$default = (string) get_option($field['id'], $field['default']);

			$i = 0;
			foreach ($field['options'] as $key => $value) {
				$dataAtts = '';
				if (!empty($field['toggle_target'])) {
					$dataAtts .= ' data-toggle-target=".toggle-by-' . esc_attr($field['toggle_target']) . '" data-toggle-value="' . esc_attr($key) . '"';
				}

				echo '<label for="' . esc_attr($field['id']) . '-' . ++$i . '" class="wpsseo-radio-default"><input name="' . esc_attr($field['id']) . '" id="' . esc_attr($field['id']) . '-' . $i . '" type="' . esc_attr($field['type']) . '" value="' . esc_attr($key) . '"' . $dataAtts . ((string) $key === $default ? ' checked' : '') . '><span class="check"></span> ' . esc_html($value) . '</label>';
			}
		}

		if (!empty($field['description'])) {
			echo '<p class="description" id="' . esc_attr($field['id']) . '-description">' . wp_kses_post($field['description']) . '</p>';
		}

		echo '</fieldset>';
	}

	/**
     * @param array<string, mixed> $field Field configuration
     */
    private function renderColorField(array $field): void
	{
		$default = get_option($field['id'], $field['default']);

		echo '<input name="' . esc_attr($field['id']) . '" id="' . esc_attr($field['id']) . '" type="text" value="' . esc_attr($default) . '" data-default-color="' . esc_attr($default) . '" class="regular-text color-picker">';

		if (!empty($field['description'])) {
			echo '<p class="description" id="' . esc_attr($field['id']) . '-description">' . wp_kses_post($field['description']) . '</p>';
		}
	}

	/**
     * @param array<string, mixed> $field Field configuration
     */
    private function renderTextareaField(array $field): void
	{
		$default = get_option($field['id'], $field['default']);
		$size = $field['size'] ?? 'regular';
		$class = $size . '-text';
		$rows = $size === 'large' ? 12 : 6;

		echo '<textarea name="' . esc_attr($field['id']) . '" id="' . esc_attr($field['id']) . '" rows="' . esc_attr((string) $rows) . '" class="' . esc_attr($class) . ' code">' . esc_textarea($default) . '</textarea>';

		if (!empty($field['description'])) {
			echo '<p class="description" id="' . esc_attr($field['id']) . '-description">' . wp_kses_post($field['description']) . '</p>';
		}
	}

	/**
     * @param array<string, mixed> $field Field configuration
     */
    private function renderNumberField(array $field): void
	{
		$default = get_option($field['id'], $field['default']);

		$min = $field['min'] ?? '0';
		$max = $field['max'] ?? null;
		$step = $field['step'] ?? '1';

		echo '<input name="' . esc_attr($field['id']) . '" id="' . esc_attr($field['id']) . '" type="' . esc_attr($field['type']) . '" value="' . esc_attr($default) . '" min="' . esc_attr($min) . '"' . (empty($max) ? '' : ' max="' . esc_attr($max) . '"') . ' step="' . esc_attr($step) . '" class="regular-text">';

		if (!empty($field['description'])) {
			echo '<p class="description" id="' . esc_attr($field['id']) . '-description">' . wp_kses_post($field['description']) . '</p>';
		}
	}

	/**
     * @param array<string, mixed> $field Field configuration
     */
    private function renderPasswordField(array $field): void
	{
		$default = esc_attr(get_option($field['id']));
		$placeholder = $field['placeholder'] ?? '';
		$purgeButton = (bool) ($field['purge_button'] ?? true);
		$purgeButtonText = $field['purge_button_text'] ?? __('Purge', Constants::TEXT_DOMAIN);
		$confirmDialog = $field['confirm_dialog'] ?? __('Are you sure?', Constants::TEXT_DOMAIN);

		$params = '';
		if (!empty($default)) {
			$default = str_repeat('*', mb_strlen($default));
			$params = ' readonly="readonly" style="pointer-events: none"';
		}

		echo '<div class="wpsseo-password-wrapper">';
		echo '<input name="' . esc_attr($field['id']) . '" id="' . esc_attr($field['id']) . '" type="' . esc_attr($field['type']) . '" value="' . esc_attr($default) . '" placeholder="' . esc_attr($placeholder) . '" class="regular-text" autocomplete="off"' . $params . '>';

		if (!empty($default) && $purgeButton) {
			echo '<a href="#" class="wpsseo-button purge" data-confirm="' . esc_attr($confirmDialog) . '">' . esc_html($purgeButtonText) . '</a>';
		}

		echo '</div>';

		if (!empty($field['description'])) {
			echo '<p class="description" id="' . esc_attr($field['id']) . '-description">' . wp_kses_post($field['description']) . '</p>';
		}
	}

	/**
     * @param array<string, mixed> $field Field configuration
     */
    private function renderEditorField(array $field): void
	{
		$default = get_option($field['id']);

		wp_editor($default, esc_attr($field['id']));

		if (!empty($field['description'])) {
			echo '<p class="description" id="' . esc_attr($field['id']) . '-description">' . wp_kses_post($field['description']) . '</p>';
		}
	}

	/**
     * @param array<string, mixed> $field Field configuration
     */
    private function renderMessageField(array $field): void
	{
		if (!empty($field['description'])) {
			echo '<div class="description">' . wp_kses_post($field['description']) . '</div>';
		}
	}

	/**
     * @param array<string, mixed> $field Field configuration
     */
    private function renderTextGroupField(array $field): void
	{
		echo '<fieldset>';

		if (!empty($field['options'])) {
			$saved = get_option($field['id'], []);

			$i = 0;
			foreach ($field['options'] as $key => $value) {
				$default = $saved[$key] ?? '';

				if (++$i > 1) {
					echo '<br>';
				}

				echo '<label for="' . esc_attr($field['id']) . '-' . $i . '" class="text-group">' . esc_html($value) . '</label>';
				echo '<input name="' . esc_attr($field['id']) . '[' . esc_attr($key) . ']" id="' . esc_attr($field['id']) . '-' . $i . '" type="text" value="' . esc_attr($default) . '" class="regular-text">';
			}
		}

		if (!empty($field['description'])) {
			echo '<p class="description" id="' . esc_attr($field['id']) . '-description">' . wp_kses_post($field['description']) . '</p>';
		}

		echo '</fieldset>';
	}

	/**
     * @param array<string, mixed> $field Field configuration
     */
    private function renderTextField(array $field): void
	{
		$default = get_option($field['id'], $field['default']);

		echo '<input name="' . esc_attr($field['id']) . '" id="' . esc_attr($field['id']) . '" type="' . esc_attr($field['type']) . '" value="' . esc_attr($default) . '" class="regular-text">';

		if (!empty($field['description'])) {
			echo '<p class="description" id="' . esc_attr($field['id']) . '-description">' . wp_kses_post($field['description']) . '</p>';
		}
	}
}
