<?php

declare(strict_types=1);

namespace SignocoreToolkit\Admin;

use SignocoreToolkit\Application\Constants;
use SignocoreToolkit\Infrastructure\Traits\Singleton;
use SignocoreToolkit\Support\PluginCompatibility;

/**
 * Branded admin notice recommending Signocore SEO when not active.
 *
 * @package SignocoreToolkit\Admin
 * @since 3.1.0
 */
final class SeoRecommendation
{
	use Singleton;

	private const NOTICE_ID = 'sctk_seo_recommendation';

	private const META_KEY = 'sctk_dismissed_seo_recommendation';

	private const PURCHASE_URL = 'https://signocore.com/wordpress-plugins/signocore-seo/';

	protected function init(): void
	{
		add_action('admin_notices', [$this, 'render']);
		add_action('wp_ajax_sctk_dismiss_seo_recommendation', [$this, 'handleDismiss']);
	}

	/**
	 * Render the branded recommendation notice.
	 */
	public function render(): void
	{
		if (defined('SIGNOCORE_SEO_FILE')) {
			return;
		}

		if (!current_user_can('manage_options')) {
			return;
		}

		if ($this->isDismissed()) {
			return;
		}

		$isInstalled = PluginCompatibility::isSeoInstalled();

		if ($isInstalled) {
			$this->renderActivateNotice();
		} else {
			$this->renderPromotionalNotice();
		}

		$this->renderDismissScript();
	}

	/**
	 * Render notice when signocore-seo is not installed.
	 */
	private function renderPromotionalNotice(): void
	{
		$purchaseUrl = esc_url(self::PURCHASE_URL);

		?>
		<div id="<?php echo esc_attr(self::NOTICE_ID); ?>" class="notice sctk-seo-notice" style="
			display: flex;
			align-items: flex-start;
			gap: 16px;
			padding: 16px 20px;
			background: linear-gradient(135deg, #fff 0%, #faf8f6 100%);
			border: 1px solid #e2e8f0;
			border-left: 4px solid #b8642a;
			border-radius: 4px;
			box-shadow: 0 1px 2px rgba(0, 0, 0, 0.05);
			position: relative;
		">
			<div style="
				flex-shrink: 0;
				width: 48px;
				height: 48px;
				background: #546e7a;
				border-radius: 8px;
				display: flex;
				align-items: center;
				justify-content: center;
				margin-top: 2px;
			">
				<svg width="28" height="28" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
					<path d="M12 2L15.09 8.26L22 9.27L17 14.14L18.18 21.02L12 17.77L5.82 21.02L7 14.14L2 9.27L8.91 8.26L12 2Z" fill="#fff" stroke="#fff" stroke-width="1" stroke-linecap="round" stroke-linejoin="round"/>
				</svg>
			</div>
			<div style="flex: 1; min-width: 0;">
				<h3 style="margin: 0 0 6px; font-size: 15px; font-weight: 600; color: #1e293b;">
					<?php esc_html_e('Unlock the full potential of Signocore Toolkit', Constants::TEXT_DOMAIN); ?>
				</h3>
				<p style="margin: 0 0 12px; color: #475569; font-size: 13px; line-height: 1.5;">
					<?php esc_html_e('Add Signocore SEO to get automatic meta tags, schema markup, XML sitemaps, and instant search engine indexing via IndexNow. Works seamlessly with Signocore Toolkit for a complete WordPress optimization suite.', Constants::TEXT_DOMAIN); ?>
				</p>
				<a href="<?php echo $purchaseUrl; ?>" target="_blank" rel="noopener" style="
					display: inline-flex;
					align-items: center;
					gap: 6px;
					background: #b8642a;
					color: #fff;
					padding: 7px 16px;
					border-radius: 4px;
					font-size: 13px;
					font-weight: 600;
					text-decoration: none;
					border: 1px solid #9e5524;
					box-shadow: 0 1px 2px rgba(0, 0, 0, 0.05);
					transition: background 150ms ease;
				"
				onmouseover="this.style.background='#9e5524'"
				onmouseout="this.style.background='#b8642a'"
				>
					<?php esc_html_e('Get Signocore SEO', Constants::TEXT_DOMAIN); ?>
					<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M18 13v6a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h6"/><polyline points="15 3 21 3 21 9"/><line x1="10" y1="14" x2="21" y2="3"/></svg>
				</a>
			</div>
			<button type="button" class="sctk-seo-dismiss" aria-label="<?php esc_attr_e('Dismiss', Constants::TEXT_DOMAIN); ?>" style="
				position: absolute;
				top: 8px;
				right: 8px;
				background: none;
				border: none;
				cursor: pointer;
				padding: 4px;
				color: #94a3b8;
				line-height: 1;
				transition: color 150ms ease;
			"
			onmouseover="this.style.color='#475569'"
			onmouseout="this.style.color='#94a3b8'"
			>
				<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><line x1="18" y1="6" x2="6" y2="18"/><line x1="6" y1="6" x2="18" y2="18"/></svg>
			</button>
		</div>
		<?php
	}

	/**
	 * Render notice when signocore-seo is installed but not activated.
	 */
	private function renderActivateNotice(): void
	{
		$pluginFile = 'signocore-seo/signocore-seo.php';
		$activateUrl = esc_url(wp_nonce_url(admin_url('plugins.php?action=activate&plugin=' . $pluginFile), 'activate-plugin_' . $pluginFile));

		?>
		<div id="<?php echo esc_attr(self::NOTICE_ID); ?>" class="notice sctk-seo-notice" style="
			display: flex;
			align-items: flex-start;
			gap: 16px;
			padding: 16px 20px;
			background: linear-gradient(135deg, #fff 0%, #f6faf8 100%);
			border: 1px solid #e2e8f0;
			border-left: 4px solid #15803d;
			border-radius: 4px;
			box-shadow: 0 1px 2px rgba(0, 0, 0, 0.05);
			position: relative;
		">
			<div style="
				flex-shrink: 0;
				width: 48px;
				height: 48px;
				background: #546e7a;
				border-radius: 8px;
				display: flex;
				align-items: center;
				justify-content: center;
				margin-top: 2px;
			">
				<svg width="28" height="28" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
					<path d="M22 11.08V12a10 10 0 1 1-5.93-9.14" stroke="#fff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" fill="none"/>
					<polyline points="22 4 12 14.01 9 11.01" stroke="#fff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" fill="none"/>
				</svg>
			</div>
			<div style="flex: 1; min-width: 0;">
				<h3 style="margin: 0 0 6px; font-size: 15px; font-weight: 600; color: #1e293b;">
					<?php esc_html_e('Signocore SEO is ready to activate', Constants::TEXT_DOMAIN); ?>
				</h3>
				<p style="margin: 0 0 12px; color: #475569; font-size: 13px; line-height: 1.5;">
					<?php esc_html_e('Signocore SEO is installed but not yet activated. Activate it to enable automatic meta tags, structured data, XML sitemaps, and more. Together with Signocore Toolkit, you get the complete WordPress optimization experience.', Constants::TEXT_DOMAIN); ?>
				</p>
				<a href="<?php echo $activateUrl; ?>" style="
					display: inline-flex;
					align-items: center;
					gap: 6px;
					background: #15803d;
					color: #fff;
					padding: 7px 16px;
					border-radius: 4px;
					font-size: 13px;
					font-weight: 600;
					text-decoration: none;
					border: 1px solid #116932;
					box-shadow: 0 1px 2px rgba(0, 0, 0, 0.05);
					transition: background 150ms ease;
				"
				onmouseover="this.style.background='#116932'"
				onmouseout="this.style.background='#15803d'"
				>
					<?php esc_html_e('Activate Signocore SEO', Constants::TEXT_DOMAIN); ?>
					<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="9 18 15 12 9 6"/></svg>
				</a>
			</div>
			<button type="button" class="sctk-seo-dismiss" aria-label="<?php esc_attr_e('Dismiss', Constants::TEXT_DOMAIN); ?>" style="
				position: absolute;
				top: 8px;
				right: 8px;
				background: none;
				border: none;
				cursor: pointer;
				padding: 4px;
				color: #94a3b8;
				line-height: 1;
				transition: color 150ms ease;
			"
			onmouseover="this.style.color='#475569'"
			onmouseout="this.style.color='#94a3b8'"
			>
				<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><line x1="18" y1="6" x2="6" y2="18"/><line x1="6" y1="6" x2="18" y2="18"/></svg>
			</button>
		</div>
		<?php
	}

	/**
	 * Render inline JavaScript for AJAX dismissal.
	 */
	private function renderDismissScript(): void
	{
		$nonce = wp_create_nonce('sctk_dismiss_seo_recommendation');
		$noticeId = esc_js(self::NOTICE_ID);

		?>
		<script>
		(function() {
			var notice = document.getElementById('<?php echo $noticeId; ?>');
			if (!notice) return;

			notice.querySelector('.sctk-seo-dismiss').addEventListener('click', function() {
				notice.style.transition = 'opacity 200ms ease, max-height 200ms ease';
				notice.style.opacity = '0';
				notice.style.maxHeight = notice.offsetHeight + 'px';
				setTimeout(function() {
					notice.style.maxHeight = '0';
					notice.style.margin = '0';
					notice.style.padding = '0';
					notice.style.overflow = 'hidden';
				}, 100);
				setTimeout(function() { notice.remove(); }, 300);

				var xhr = new XMLHttpRequest();
				xhr.open('POST', ajaxurl);
				xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
				xhr.send('action=sctk_dismiss_seo_recommendation&nonce=<?php echo esc_js($nonce); ?>');
			});
		})();
		</script>
		<?php
	}

	/**
	 * Handle AJAX dismiss request.
	 */
	public function handleDismiss(): void
	{
		if (!check_ajax_referer('sctk_dismiss_seo_recommendation', 'nonce', false)) {
			wp_send_json_error();
		}

		$userId = get_current_user_id();

		if ($userId === 0) {
			wp_send_json_error();
		}

		update_user_meta($userId, self::META_KEY, true);
		wp_send_json_success();
	}

	/**
	 * Check if current user has dismissed this notice.
	 */
	private function isDismissed(): bool
	{
		$userId = get_current_user_id();

		if ($userId === 0) {
			return false;
		}

		return (bool) get_user_meta($userId, self::META_KEY, true);
	}
}
