<?php

declare(strict_types=1);

namespace SignocoreToolkit\Admin;

use SignocoreToolkit\Infrastructure\Traits\Singleton;
use SignocoreToolkit\Application\Constants;

final class AdminTweaks
{
	use Singleton;

	protected function init(): void
	{
		add_action('admin_init', [$this, 'optimizeWp']);
		add_action('admin_init', [$this, 'addEnvironmentSettings']);
		add_action('admin_notices', [$this, 'optimizeNotice']);
		add_action('admin_bar_menu', [$this, 'customizeToolbar'], 999);
		add_action('admin_menu', [$this, 'changePostMenuLabel']);
		add_action('admin_head', [$this, 'changePostLabel']);
	}

	public function customizeToolbar(\WP_Admin_Bar $wpAdminBar): void
	{
		if (!Constants::$isAdmin) {
			$wpAdminBar->remove_node('search');
		}

		if (Constants::$isAdmin && current_user_can('manage_options')) {
			// Add nonce for CSRF protection
			$optimizeUrl = wp_nonce_url(
				admin_url('/?signocore-optimize-wp'),
				'signocore_optimize_wp_action',
				'signocore_optimize_wp_nonce'
			);

			$args = [
				'id'	=> 'optimize-wp',
				'title'	=> '<span class="ab-icon dashicons dashicons-superhero" style="line-height: 1.2"></span>' . esc_html__('Optimize WP', Constants::TEXT_DOMAIN),
				'href'	=> $optimizeUrl,
				'meta'	=> [
					'title' => __('Perform the following tasks:

- Clean old revisions (keep 5)
- Remove spam comments
- Optimize database tables
- Clear known server caches
- Clear object cache
- Clear known plugin caches
- Clear all transients
- Flush rewrite rules', Constants::TEXT_DOMAIN)
				],
			];

			$wpAdminBar->add_node($args);
		}
	}

	/**
     * Perform WordPress optimization tasks including revision cleanup, spam removal, and cache clearing.
     *
     * Security:
     * - Requires 'manage_options' capability
     * - Verifies nonce to prevent CSRF attacks
     */
    public function optimizeWp(): void
	{
		$optimizeWp = isset($_GET['signocore-optimize-wp']);
		if (!$optimizeWp) {
			return;
		}

		// Verify user capability
		if (!current_user_can('manage_options')) {
			wp_die(
				esc_html__('You do not have sufficient permissions to perform this action.', Constants::TEXT_DOMAIN),
				esc_html__('Permission Denied', Constants::TEXT_DOMAIN),
				['response' => 403]
			);
		}

		// Verify nonce for CSRF protection
		$nonce = sanitize_text_field(wp_unslash($_GET['signocore_optimize_wp_nonce'] ?? ''));
		if (!wp_verify_nonce($nonce, 'signocore_optimize_wp_action')) {
			wp_die(
				esc_html__('Security check failed. Please try again.', Constants::TEXT_DOMAIN),
				esc_html__('Security Error', Constants::TEXT_DOMAIN),
				['response' => 403]
			);
		}

		global $wpdb;

		// Track optimization stats for logging
		$stats = [
			'revisions_deleted' => 0,
			'spam_deleted' => 0,
			'transients_deleted' => 0,
			'tables_optimized' => 0,
		];

		// Remove Old Post Revisions (batched for performance)
		$results = $wpdb->get_results(
			$wpdb->prepare(
				sprintf('SELECT `ID`, `post_parent` FROM `%s` WHERE `post_type` = %%s ORDER BY `post_parent`, `ID` ASC', $wpdb->posts),
				'revision'
			)
		);

		$revisions = [];
		foreach ($results as $result) {
			$revisions[$result->post_parent][] = $result->ID;
		}

		$revisionsToKeep = 5;
		foreach ($revisions as $parent => $ids) {
			$sliced = array_slice($ids, 0, -$revisionsToKeep);

			if ($sliced === []) {
				unset($revisions[$parent]);
			} else {
				$revisions[$parent] = $sliced;
			}
		}

		if ($revisions !== []) {
			$idsToDelete = array_merge(...$revisions);
			$batchSize = 50; // Process 50 revisions at a time to prevent timeout

			// Process in batches
			foreach (array_chunk($idsToDelete, $batchSize) as $batch) {
				foreach ($batch as $id) {
					$deleted = wp_delete_post_revision($id);
					if ($deleted) {
						$stats['revisions_deleted']++;
					}
				}

				// Prevent memory exhaustion on large operations
				wp_cache_flush();
			}
		}

		// Remove Spam Comments (batched)
		$spamComments = get_comments([
			'fields' => 'ids',
			'status' => 'spam',
			'number' => 1000, // Limit to prevent timeout
		]);

		if (is_array($spamComments) && $spamComments !== []) {
			$batchSize = 50;

			foreach (array_chunk($spamComments, $batchSize) as $batch) {
				foreach ($batch as $commentId) {
					$deleted = wp_delete_comment($commentId, true);
					if ($deleted) {
						$stats['spam_deleted']++;
					}
				}
			}
		}

		// Remove Transients
		$transientCount = $wpdb->query(
			$wpdb->prepare(
				sprintf('DELETE FROM `%s` WHERE `option_name` LIKE %%s OR `option_name` LIKE %%s', $wpdb->options),
				$wpdb->esc_like('_transient_') . '%',
				$wpdb->esc_like('_transient_timeout_') . '%'
			)
		);
		$stats['transients_deleted'] = (int) $transientCount;

		// Optimize Database Tables
		$tables = $wpdb->get_results("SHOW TABLE STATUS");
		foreach ($tables as $table) {
			if ((int) $table->Data_free > 0) {
				// Validate table name to prevent SQL injection
				$tableName = preg_replace('/[^a-zA-Z0-9_]/', '', $table->Name);
				if ($tableName === $table->Name && str_starts_with($tableName, $wpdb->prefix)) {
					$optimized = $wpdb->query(sprintf('OPTIMIZE TABLE `%s`', $tableName));
					if ($optimized !== false) {
						$stats['tables_optimized']++;
					}
				}
			}
		}

		// Clear SpinupWP Cache
		if (function_exists('spinupwp_purge_site')) {
			spinupwp_purge_site();
		}

		// Clear OPCache
		if (function_exists('opcache_reset')) {
			opcache_reset();
		}

		// Clear WP object cache
		wp_cache_flush();

		// Clear WP Super Cache
		if (function_exists('wp_cache_clear_cache')) {
			wp_cache_clear_cache();
		}

		// Clear WP Rocket Cache
		if (function_exists('rocket_clean_domain')) {
			rocket_clean_domain();
		}

		// Clear WP Fastest Cache
		if (function_exists('wpfc_clear_all_cache')) {
			wpfc_clear_all_cache();
		}

		// Clear W3 Total Cache
		if (function_exists('w3tc_pgcache_flush')) {
			w3tc_pgcache_flush();
		}

		// Clear WP-Optimize Cache
		if (function_exists('wp_optimize_clear_cache')) {
			wp_optimize_clear_cache();
		}

		// Clear WP-FFPC Cache
		if (function_exists('wp_ffpc_flush_all')) {
			wp_ffpc_flush_all();
		}

		// Flush permalinks
		flush_rewrite_rules();

		// Store stats in transient for display in admin notice
		set_transient('sctk_optimization_stats', $stats, 60);

		$referer = add_query_arg('signocore-wp-optimized', 'true', esc_url_raw($_SERVER['HTTP_REFERER'] ?? home_url('/')));
		wp_redirect($referer);

		exit;
	}

	/**
     * Display optimization success notice with statistics.
     */
    public function optimizeNotice(): void
	{
		$optimized = (bool) ($_GET['signocore-wp-optimized'] ?? false);

		if (!$optimized) {
			return;
		}

		// Retrieve optimization stats
		$stats = get_transient('sctk_optimization_stats');
		delete_transient('sctk_optimization_stats');

		$message = esc_html__('WordPress was optimized successfully!', Constants::TEXT_DOMAIN);

		// Add stats if available
		if ($stats && is_array($stats)) {
			$details = [];

			if ($stats['revisions_deleted'] > 0) {
				$details[] = sprintf(
					/* translators: %d: number of revisions deleted */
					_n('%d revision deleted', '%d revisions deleted', $stats['revisions_deleted'], Constants::TEXT_DOMAIN),
					$stats['revisions_deleted']
				);
			}

			if ($stats['spam_deleted'] > 0) {
				$details[] = sprintf(
					/* translators: %d: number of spam comments deleted */
					_n('%d spam comment removed', '%d spam comments removed', $stats['spam_deleted'], Constants::TEXT_DOMAIN),
					$stats['spam_deleted']
				);
			}

			if ($stats['transients_deleted'] > 0) {
				$details[] = sprintf(
					/* translators: %d: number of transients deleted */
					_n('%d transient cleared', '%d transients cleared', $stats['transients_deleted'], Constants::TEXT_DOMAIN),
					$stats['transients_deleted']
				);
			}

			if ($stats['tables_optimized'] > 0) {
				$details[] = sprintf(
					/* translators: %d: number of database tables optimized */
					_n('%d table optimized', '%d tables optimized', $stats['tables_optimized'], Constants::TEXT_DOMAIN),
					$stats['tables_optimized']
				);
			}

			if ($details !== []) {
				$message .= ' <strong>' . implode(', ', $details) . '.</strong>';
			}
		}

		echo '<div class="notice notice-success signocore-notice is-dismissible">
			<p>' . $message . '</p>
		</div>';
	}

	public function addEnvironmentSettings(): void
	{
		$args = [
			'type' => 'string',
			'sanitize_callback' => 'sanitize_text_field',
			'default' => null,
		];

		register_setting(
			'reading',
			'signocore_environment',
			$args
		);

		add_settings_field(
			'signocore_environment',
			__('Environment', Constants::TEXT_DOMAIN),
			[$this, 'renderEnvironmentField'],
			'reading',
			'default',
			['label_for' => 'signocore_environment'],
		);
	}

	public function renderEnvironmentField(): void
    {
        $currentEnvironment = get_option('signocore_environment', 'development');
        $availableEnvironments = [
			'production' => _x('Production', 'Environment', Constants::TEXT_DOMAIN),
			'local' => _x('Local', 'Environment', Constants::TEXT_DOMAIN),
			'development' => _x('Development', 'Environment', Constants::TEXT_DOMAIN),
			'staging' => _x('Staging', 'Environment', Constants::TEXT_DOMAIN),
		];
        echo '<select id="signocore_environment" name="signocore_environment">';
        foreach ($availableEnvironments as $value => $label) {
			echo '<option value="' . esc_attr($value) . '" ' . ($value === $currentEnvironment ? 'selected="selected"' : '') . '>' . esc_html($label) . '</option>';
		}

        echo '</select>';
    }

	public function changePostMenuLabel(): void
	{
		global $menu;
		$menu[5][0] = __('Blog Posts', Constants::TEXT_DOMAIN);
	}

	public function changePostLabel(): void
	{
		/** @var \WP_Post_Type|null $post_type_object */
		global $post_type_object;

		if ($post_type_object instanceof \WP_Post_Type && $post_type_object->name === 'post') {
			$post_type_object->labels->name = __('Blog Posts', Constants::TEXT_DOMAIN);
			$post_type_object->labels->name_admin_bar = __('Blog Post', Constants::TEXT_DOMAIN);
		}
	}
}
