<?php

declare(strict_types=1);

namespace SignocoreToolkit\Admin;

use SignocoreToolkit\Infrastructure\Traits\Singleton;
use SignocoreToolkit\Application\Constants;
use SignocoreToolkit\Application\OptionsManager;

/**
 * Persistent admin notification system with dismissal tracking.
 *
 * @package SignocoreToolkit\Admin
 * @since 1.0.0
 */
final class AdminNotices
{
	use Singleton;

	/** Option key for storing persistent notices */
	private const NOTICES_OPTION_KEY = 'admin_notices';

	/** User meta key prefix for dismissed notices */
	private const DISMISSED_META_PREFIX = 'dismissed_notice_';

	/**
	 * Transient notices (shown once).
	 *
	 * @var array<string, array<string, mixed>>
	 */
	private static array $transientNotices = [];

	/**
     * Initialize admin notices hooks.
     */
    protected function init(): void
	{
		add_action('admin_notices', [$this, 'displayNotices']);
		add_action('wp_ajax_signocore_toolkit_dismiss_notice', [$this, 'handleDismissAjax']);
		add_action('admin_enqueue_scripts', [$this, 'enqueueScripts']);
	}

	/**
     * Add a success notice.
     *
     * @param string $message Notice message
     * @param string|null $id Unique notice ID
     * @param bool $persistent Whether notice persists across page loads
     * @param bool $dismissible Whether notice can be dismissed
     */
    public static function success(
		string $message,
		?string $id = null,
		bool $persistent = false,
		bool $dismissible = true
	): void {
		self::add($message, $id, 'success', $persistent, $dismissible);
	}

	/**
     * Add an error notice.
     *
     * @param string $message Notice message
     * @param string|null $id Unique notice ID
     * @param bool $persistent Whether notice persists across page loads
     * @param bool $dismissible Whether notice can be dismissed
     */
    public static function error(
		string $message,
		?string $id = null,
		bool $persistent = false,
		bool $dismissible = true
	): void {
		self::add($message, $id, 'error', $persistent, $dismissible);
	}

	/**
     * Add a warning notice.
     *
     * @param string $message Notice message
     * @param string|null $id Unique notice ID
     * @param bool $persistent Whether notice persists across page loads
     * @param bool $dismissible Whether notice can be dismissed
     */
    public static function warning(
		string $message,
		?string $id = null,
		bool $persistent = false,
		bool $dismissible = true
	): void {
		self::add($message, $id, 'warning', $persistent, $dismissible);
	}

	/**
     * Add an info notice.
     *
     * @param string $message Notice message
     * @param string|null $id Unique notice ID
     * @param bool $persistent Whether notice persists across page loads
     * @param bool $dismissible Whether notice can be dismissed
     */
    public static function info(
		string $message,
		?string $id = null,
		bool $persistent = false,
		bool $dismissible = true
	): void {
		self::add($message, $id, 'info', $persistent, $dismissible);
	}

	/**
     * Add a custom notice.
     *
     * @param string $message Notice message
     * @param string|null $id Unique notice ID
     * @param string $type Notice type (success, error, warning, info)
     * @param bool $persistent Whether notice persists across page loads
     * @param bool $dismissible Whether notice can be dismissed
     */
    public static function add(
		string $message,
		?string $id = null,
		string $type = 'info',
		bool $persistent = false,
		bool $dismissible = true
	): void {
		if ($id === null) {
			$id = 'notice_' . md5($message . time());
		}

		$notice = [
			'id' => $id,
			'message' => $message,
			'type' => $type,
			'dismissible' => $dismissible,
			'timestamp' => time(),
		];

		if ($persistent) {
			self::addPersistentNotice($notice);
		} else {
			self::$transientNotices[$id] = $notice;
		}
	}

	/**
     * Display all active notices.
     */
    public function displayNotices(): void
	{
		$currentUserId = get_current_user_id();

		$persistentNotices = self::getPersistentNotices();

		$allNotices = array_merge($persistentNotices, self::$transientNotices);

		foreach ($allNotices as $notice) {
			if ($this->isNoticeDismissed($notice['id'], $currentUserId)) {
				continue;
			}

			$this->renderNotice($notice);
		}

		self::$transientNotices = [];
	}

	/**
     * Render a single notice.
     *
     * @param array<string, mixed> $notice Notice data
     */
    private function renderNotice(array $notice): void
	{
		$classes = ['notice', 'notice-' . $notice['type']];

		if ($notice['dismissible']) {
			$classes[] = 'is-dismissible';
			$classes[] = 'signocore-toolkit-notice';
		}

		$classString = implode(' ', $classes);
		$noticeId = esc_attr($notice['id']);
		$message = wp_kses_post($notice['message']);

		echo '<div class="' . esc_attr($classString) . '" data-notice-id="' . $noticeId . '">';
		echo '<p>' . $message . '</p>';
		echo '</div>';
	}

	/**
     * Add notice to persistent storage.
     *
     * @param array<string, mixed> $notice Notice data
     */
    private static function addPersistentNotice(array $notice): void
	{
		$notices = self::getPersistentNotices();
		$notices[$notice['id']] = $notice;

		OptionsManager::set(self::NOTICES_OPTION_KEY, $notices);
	}

	/**
	 * Get all persistent notices.
	 *
	 * @return array<string, array<string, mixed>>
	 */
	private static function getPersistentNotices(): array
	{
		return OptionsManager::getArray(self::NOTICES_OPTION_KEY, []);
	}

	/**
	 * Remove a persistent notice.
	 *
	 * @param string $id Notice ID
	 * @return bool
	 */
	public static function remove(string $id): bool
	{
		$notices = self::getPersistentNotices();

		if (!isset($notices[$id])) {
			return false;
		}

		unset($notices[$id]);
		return OptionsManager::set(self::NOTICES_OPTION_KEY, $notices);
	}

	/**
	 * Remove all persistent notices.
	 *
	 * @return bool
	 */
	public static function removeAll(): bool
	{
		return OptionsManager::set(self::NOTICES_OPTION_KEY, []);
	}

	/**
	 * Dismiss a notice for a user.
	 *
	 * @param string $id Notice ID
	 * @param int|null $userId User ID (null for current user)
	 * @return bool
	 */
	public static function dismiss(string $id, ?int $userId = null): bool
	{
		if ($userId === null) {
			$userId = get_current_user_id();
		}

		if ($userId === 0) {
			return false;
		}

		return update_user_meta($userId, self::DISMISSED_META_PREFIX . $id, true) !== false;
	}

	/**
	 * Check if a notice has been dismissed by user.
	 *
	 * @param string $id Notice ID
	 * @param int $userId User ID
	 * @return bool
	 */
	private function isNoticeDismissed(string $id, int $userId): bool
	{
		if ($userId === 0) {
			return false;
		}

		return (bool) get_user_meta($userId, self::DISMISSED_META_PREFIX . $id, true);
	}

	/**
	 * Un-dismiss a notice for a user.
	 *
	 * @param string $id Notice ID
	 * @param int|null $userId User ID (null for current user)
	 * @return bool
	 */
	public static function undismiss(string $id, ?int $userId = null): bool
	{
		if ($userId === null) {
			$userId = get_current_user_id();
		}

		if ($userId === 0) {
			return false;
		}

		return delete_user_meta($userId, self::DISMISSED_META_PREFIX . $id);
	}

	/**
     * Handle AJAX dismiss request.
     */
    public function handleDismissAjax(): void
	{
		if (!check_ajax_referer('signocore-toolkit-dismiss-notice', 'nonce', false)) {
			wp_send_json_error(['message' => 'Invalid nonce']);
		}

		$noticeId = isset($_POST['notice_id']) ? sanitize_text_field(wp_unslash($_POST['notice_id'])) : '';

		if (empty($noticeId)) {
			wp_send_json_error(['message' => 'Missing notice ID']);
		}

		if (self::dismiss($noticeId)) {
			wp_send_json_success(['message' => 'Notice dismissed']);
		} else {
			wp_send_json_error(['message' => 'Failed to dismiss notice']);
		}
	}

	/**
     * Enqueue JavaScript for AJAX dismissal.
     */
    public function enqueueScripts(): void
	{
		wp_add_inline_script(
			'common',
			"
			jQuery(document).ready(function($) {
				$(document).on('click', '.signocore-toolkit-notice .notice-dismiss', function() {
					var noticeId = $(this).closest('.signocore-toolkit-notice').data('notice-id');

					$.ajax({
						url: ajaxurl,
						type: 'POST',
						data: {
							action: 'signocore_toolkit_dismiss_notice',
							notice_id: noticeId,
							nonce: '" . wp_create_nonce('signocore-toolkit-dismiss-notice') . "'
						}
					});
				});
			});
			"
		);
	}

	/**
	 * Get all active notices for current user.
	 *
	 * @return array<string, array<string, mixed>>
	 */
	public static function getActiveNotices(): array
	{
		$userId = get_current_user_id();
		$notices = self::getPersistentNotices();
		$active = [];

		foreach ($notices as $id => $notice) {
			if (!get_user_meta($userId, self::DISMISSED_META_PREFIX . $id, true)) {
				$active[$id] = $notice;
			}
		}

		return $active;
	}

	/**
	 * Clean up dismissed notice meta for notices that no longer exist.
	 *
	 * @return int Number of meta entries removed
	 */
	public static function cleanupDismissedMeta(): int
	{
		global $wpdb;

		$persistentNoticeIds = array_keys(self::getPersistentNotices());
		$metaKeys = [];

		foreach ($persistentNoticeIds as $id) {
			$metaKeys[] = self::DISMISSED_META_PREFIX . $id;
		}

		$placeholders = implode(', ', array_fill(0, count($metaKeys), '%s'));

		return (int) $wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$wpdb->usermeta}
				WHERE meta_key LIKE %s
				AND meta_key NOT IN ({$placeholders})",
				array_merge(
					[self::DISMISSED_META_PREFIX . '%'],
					$metaKeys
				)
			)
		);
	}
}
