/**
 * GDPR-Compliant Cookie Consent Manager
 *
 * Provides granular consent management for cookie categories
 * with gtag integration and legacy cookie migration.
 */
(() => {
	'use strict';

	// Configuration (injected from PHP via window.sctkConsentConfig)
	const CONFIG = window.sctkConsentConfig || {
		cookieName: 'sctk_consent',
		expiryDays: 365,
		showFloating: true,
		version: 1
	};

	const COOKIE_ATTRS = ';path=/;samesite=lax;secure';

	// ===========================================
	// Cookie Manager
	// ===========================================
	const CookieManager = {
		get(name) {
			const value = ('; ' + decodeURIComponent(document.cookie) + ';')
				.split('; ' + name + '=')
				.pop()
				.split(';')
				.shift();
			return value || '';
		},

		set(name, value, days = CONFIG.expiryDays) {
			const expires = new Date(Date.now() + days * 864e5).toUTCString();
			document.cookie = `${name}=${encodeURIComponent(value)};expires=${expires}${COOKIE_ATTRS}`;
		},

		delete(name) {
			document.cookie = `${name}=;expires=Thu, 01 Jan 1970 00:00:00 GMT${COOKIE_ATTRS}`;
		}
	};

	// ===========================================
	// Consent Manager
	// ===========================================
	const ConsentManager = {
		/**
		 * Get current consent state from cookie.
		 * @returns {Object|null} Consent object or null if not set
		 */
		getConsent() {
			const raw = CookieManager.get(CONFIG.cookieName);
			if (!raw) return null;

			try {
				const consent = JSON.parse(raw);
				// Validate structure
				if (typeof consent === 'object' && 'v' in consent) {
					return consent;
				}
			} catch (e) {
				// Invalid JSON, cookie is corrupted
			}
			return null;
		},

		/**
		 * Check if user has given consent (any choice made).
		 */
		hasConsented() {
			return this.getConsent() !== null;
		},

		/**
		 * Save consent choices.
		 * @param {boolean} analytics - Analytics consent
		 * @param {boolean} marketing - Marketing consent
		 * @param {string} method - How consent was given (accept_all, reject_all, custom)
		 */
		setConsent(analytics, marketing, method = 'custom') {
			const consent = {
				v: CONFIG.version,
				ts: Math.floor(Date.now() / 1000),
				analytics: analytics,
				marketing: marketing,
				method: method
			};

			CookieManager.set(CONFIG.cookieName, JSON.stringify(consent));
			this.updateGtag(consent);
			this.dispatchEvent('updated', consent);

			return consent;
		},

		/**
		 * Update gtag consent state.
		 * @param {Object} consent - Consent object
		 */
		updateGtag(consent) {
			if (typeof gtag !== 'function') return;

			const analyticsState = consent.analytics ? 'granted' : 'denied';
			const marketingState = consent.marketing ? 'granted' : 'denied';

			gtag('consent', 'update', {
				analytics_storage: analyticsState,
				ad_storage: marketingState,
				ad_user_data: marketingState,
				ad_personalization: marketingState
			});
		},

		/**
		 * Dispatch custom event for third-party integrations.
		 * @param {string} action - Event action
		 * @param {Object} detail - Event detail data
		 */
		dispatchEvent(action, detail = {}) {
			window.dispatchEvent(new CustomEvent(`sctk:consent:${action}`, { detail }));
		},

		/**
		 * Migrate from legacy cookie formats.
		 */
		migrateLegacy() {
			// Check for old cookie names
			const legacyConsent = CookieManager.get('consent');
			const legacyAccepted = CookieManager.get('cookies_accepted');
			const legacyDeclined = CookieManager.get('cookies_declined');

			if (legacyConsent || legacyAccepted || legacyDeclined) {
				// Determine what the old consent was
				const wasAccepted = legacyConsent === 'accepted' || legacyAccepted;

				// Create new consent with migrated values
				this.setConsent(wasAccepted, wasAccepted, 'legacy_migration');

				// Clean up old cookies
				CookieManager.delete('consent');
				CookieManager.delete('cookies_accepted');
				CookieManager.delete('cookies_declined');

				return true;
			}

			return false;
		}
	};

	// ===========================================
	// UI Manager
	// ===========================================
	const UI = {
		elements: {
			banner: null,
			modal: null,
			floating: null,
			analyticsToggle: null,
			marketingToggle: null
		},

		/**
		 * Cache DOM elements.
		 */
		cacheElements() {
			this.elements.banner = document.getElementById('sctk-cookie-banner');
			this.elements.modal = document.getElementById('sctk-cookie-modal');
			this.elements.floating = document.getElementById('sctk-cookie-floating');
			this.elements.analyticsToggle = document.getElementById('sctk-consent-analytics');
			this.elements.marketingToggle = document.getElementById('sctk-consent-marketing');
		},

		/**
		 * Show the consent banner with animation.
		 */
		showBanner() {
			const { banner } = this.elements;
			if (!banner) return;

			banner.style.display = 'block';
			// Trigger reflow for animation
			banner.offsetHeight;
			banner.classList.add('is-visible');

			ConsentManager.dispatchEvent('banner:shown');
		},

		/**
		 * Hide the consent banner with animation.
		 */
		hideBanner() {
			const { banner } = this.elements;
			if (!banner) return;

			banner.classList.remove('is-visible');
			setTimeout(() => {
				banner.style.display = 'none';
			}, 300);
		},

		/**
		 * Show the preferences modal.
		 */
		showPreferences() {
			const { modal } = this.elements;
			if (!modal) return;

			// Sync toggles with current consent state
			const consent = ConsentManager.getConsent();
			if (consent) {
				if (this.elements.analyticsToggle) {
					this.elements.analyticsToggle.checked = consent.analytics;
				}
				if (this.elements.marketingToggle) {
					this.elements.marketingToggle.checked = consent.marketing;
				}
			}

			modal.style.display = 'flex';
			// Trigger reflow for animation
			modal.offsetHeight;
			modal.classList.add('is-open');

			// Focus trap - focus first focusable element
			const firstFocusable = modal.querySelector('button, [href], input, select, textarea, [tabindex]:not([tabindex="-1"])');
			if (firstFocusable) firstFocusable.focus();

			ConsentManager.dispatchEvent('preferences:opened');
		},

		/**
		 * Hide the preferences modal.
		 */
		hidePreferences() {
			const { modal } = this.elements;
			if (!modal) return;

			modal.classList.remove('is-open');
			setTimeout(() => {
				modal.style.display = 'none';
			}, 300);

			ConsentManager.dispatchEvent('preferences:closed');
		},

		/**
		 * Show the floating preferences button.
		 */
		showFloatingButton() {
			const { floating } = this.elements;
			if (!floating || !CONFIG.showFloating) return;

			floating.style.display = 'flex';
			// Trigger reflow for animation
			floating.offsetHeight;
			floating.classList.add('is-visible');
		},

		/**
		 * Get current toggle states from the modal.
		 * @returns {Object} Object with analytics and marketing boolean values
		 */
		getToggleStates() {
			return {
				analytics: this.elements.analyticsToggle?.checked || false,
				marketing: this.elements.marketingToggle?.checked || false
			};
		},

		/**
		 * Set toggle states in the modal.
		 * @param {boolean} analytics
		 * @param {boolean} marketing
		 */
		setToggleStates(analytics, marketing) {
			if (this.elements.analyticsToggle) {
				this.elements.analyticsToggle.checked = analytics;
			}
			if (this.elements.marketingToggle) {
				this.elements.marketingToggle.checked = marketing;
			}
		}
	};

	// ===========================================
	// Event Handlers
	// ===========================================
	const Handlers = {
		/**
		 * Accept all cookies.
		 */
		onAcceptAll() {
			ConsentManager.setConsent(true, true, 'accept_all');
			UI.hideBanner();
			UI.hidePreferences();
			UI.showFloatingButton();
		},

		/**
		 * Reject all non-essential cookies.
		 */
		onRejectAll() {
			ConsentManager.setConsent(false, false, 'reject_all');
			UI.hideBanner();
			UI.hidePreferences();
			UI.showFloatingButton();
		},

		/**
		 * Open preferences modal.
		 */
		onCustomize() {
			UI.hideBanner();
			UI.showPreferences();
		},

		/**
		 * Save current preferences from modal.
		 */
		onSavePreferences() {
			const states = UI.getToggleStates();
			ConsentManager.setConsent(states.analytics, states.marketing, 'custom');
			UI.hidePreferences();
			UI.showFloatingButton();
		},

		/**
		 * Handle floating button click.
		 */
		onFloatingClick() {
			UI.showPreferences();
		},

		/**
		 * Handle modal backdrop/close click.
		 * @param {Event} e
		 */
		onModalClose(e) {
			if (e.target.dataset.action === 'close') {
				// If no consent exists yet, this counts as rejecting
				if (!ConsentManager.hasConsented()) {
					ConsentManager.setConsent(false, false, 'dismiss');
					UI.showFloatingButton();
				}
				UI.hidePreferences();
			}
		},

		/**
		 * Handle keyboard events.
		 * @param {KeyboardEvent} e
		 */
		onKeydown(e) {
			if (e.key === 'Escape') {
				const { modal } = UI.elements;
				if (modal && modal.classList.contains('is-open')) {
					// If no consent exists yet, this counts as rejecting
					if (!ConsentManager.hasConsented()) {
						ConsentManager.setConsent(false, false, 'dismiss');
						UI.showFloatingButton();
					}
					UI.hidePreferences();
				}
			}
		}
	};

	// ===========================================
	// Initialization
	// ===========================================
	const init = () => {
		// Check for legacy cookies and migrate
		if (ConsentManager.migrateLegacy()) {
			// Migration happened, user already has consent
			return;
		}

		// Check if consent already exists
		const existingConsent = ConsentManager.getConsent();
		if (existingConsent) {
			// Apply existing consent to gtag
			ConsentManager.updateGtag(existingConsent);

			// Show floating button when DOM is ready
			document.addEventListener('DOMContentLoaded', () => {
				UI.cacheElements();
				UI.showFloatingButton();
				bindEvents();
			});
			return;
		}

		// No consent yet - show banner when DOM is ready
		document.addEventListener('DOMContentLoaded', () => {
			UI.cacheElements();
			UI.showBanner();
			bindEvents();
		});
	};

	/**
	 * Bind event listeners to UI elements.
	 */
	const bindEvents = () => {
		// Banner buttons
		const acceptAllBtn = document.getElementById('sctk-accept-all');
		const rejectAllBtn = document.getElementById('sctk-reject-all');
		const customizeBtn = document.getElementById('sctk-customize');

		if (acceptAllBtn) acceptAllBtn.addEventListener('click', Handlers.onAcceptAll);
		if (rejectAllBtn) rejectAllBtn.addEventListener('click', Handlers.onRejectAll);
		if (customizeBtn) customizeBtn.addEventListener('click', Handlers.onCustomize);

		// Modal buttons
		const savePrefsBtn = document.getElementById('sctk-save-preferences');
		const modalAcceptBtn = document.getElementById('sctk-modal-accept-all');
		const { modal } = UI.elements;

		if (savePrefsBtn) savePrefsBtn.addEventListener('click', Handlers.onSavePreferences);
		if (modalAcceptBtn) modalAcceptBtn.addEventListener('click', Handlers.onAcceptAll);
		if (modal) modal.addEventListener('click', Handlers.onModalClose);

		// Floating button
		const { floating } = UI.elements;
		if (floating) floating.addEventListener('click', Handlers.onFloatingClick);

		// Keyboard events
		document.addEventListener('keydown', Handlers.onKeydown);
	};

	// Start initialization
	init();
})();
